package moonlight

import (
	"bytes"
	"context"
	"encoding/base64"
	"github.com/aws/aws-lambda-go/events"
	"io/ioutil"
	"net/http"
	"net/url"
	"strconv"
	"strings"
)

// ProcessRequest processes the API Gateway Proxy Request using the supplied http handler
func ProcessRequest(ctx context.Context, request events.ALBTargetGroupRequest, binaryContentTypes []string, handler http.Handler) (events.ALBTargetGroupResponse, error) {
	writer := &responseWriter{
		header: http.Header{},
	}
	req, err := generateRequest(ctx, request)
	if err != nil {
		return events.ALBTargetGroupResponse{}, err
	}

	handler.ServeHTTP(writer, req)

	return writer.getALBTargetGroupResponse(binaryContentTypes), nil
}

type responseWriter struct {
	header        http.Header
	body          []byte
	headerWritten bool
	statusCode    int
}

func (w *responseWriter) Header() http.Header {
	if w.header == nil {
		w.header = http.Header{}
	}
	return w.header
}

func (w *responseWriter) Write(bytes []byte) (int, error) {
	if !w.headerWritten {
		w.WriteHeader(http.StatusOK)
	}
	if w.body == nil {
		w.body = make([]byte, 0)
	}
	w.body = append(w.body, bytes...)
	return len(bytes), nil
}

func (w *responseWriter) WriteHeader(statusCode int) {
	if w.headerWritten {
		return
	}
	w.statusCode = statusCode
	w.Header().Set("Status", strconv.Itoa(statusCode))
	w.headerWritten = true
}

func (w *responseWriter) getALBTargetGroupResponse(binaryContentTypes []string) events.ALBTargetGroupResponse {
	resp := events.ALBTargetGroupResponse{
		StatusCode: w.statusCode,
		Headers:    make(map[string]string),
	}
	contentType := "text/plain"
	for k := range w.header {
		if strings.EqualFold("Content-Type", k) {
			contentType = w.header.Get(k)
		}
		resp.Headers[k] = w.header.Get(k)
	}

	isBinaryContentType := false

	for _, bct := range binaryContentTypes {
		if strings.EqualFold(contentType, bct) {
			isBinaryContentType = true
			break
		}
	}

	if isBinaryContentType {
		resp.Body = base64.StdEncoding.EncodeToString(w.body)
		resp.IsBase64Encoded = true
	} else {
		resp.Body = string(w.body)
	}

	return resp
}

func generateRequest(ctx context.Context, request events.ALBTargetGroupRequest) (*http.Request, error) {
	// Create a Request object that represents the call to API gateway
	req := &http.Request{
		Header: http.Header{},
		Method: request.HTTPMethod,
	}

	req = req.WithContext(ctx)

	req.URL = &url.URL{
		Host: request.Headers["Host"],
		Path: request.Path,
	}

	queryValues := req.URL.Query()
	for k, v := range request.MultiValueQueryStringParameters {
		queryValues[k] = v
	}
	req.URL.RawQuery = queryValues.Encode()

	for k, v := range request.MultiValueHeaders {
		req.Header[k] = v
	}

	for k, v := range request.Headers {
		req.Header.Set(k, v)
	}

	var bodyBytes []byte
	var err error
	if request.IsBase64Encoded {
		bodyBytes, err = base64.StdEncoding.DecodeString(request.Body)

		if err != nil {
			return nil, err
		}
	} else {
		bodyBytes = []byte(request.Body)
	}

	req.Body = ioutil.NopCloser(bytes.NewReader(bodyBytes))

	return req, nil
}
