package moonlight

import (
	//"code.justin.tv/coleiain/api-gateway-handler"
	"code.justin.tv/event-engineering/acm-ca-go/pkg/certgen"
	logging "code.justin.tv/event-engineering/golibs/pkg/logging"
	aws "code.justin.tv/event-engineering/moonlight-api/pkg/aws"
	"code.justin.tv/event-engineering/moonlight-api/pkg/ca"
	db "code.justin.tv/event-engineering/moonlight-api/pkg/db"
	types "code.justin.tv/event-engineering/moonlight-api/pkg/rpc"
	controlRPC "code.justin.tv/event-engineering/moonlight-api/pkg/rpc/control"
	daemonRPC "code.justin.tv/event-engineering/moonlight-daemon/pkg/rpc"
	"context"
	"crypto/tls"
	"fmt"
	"github.com/aws/aws-lambda-go/events"
	"github.com/twitchtv/twirp"
	"net/http"
)

type ControlAPI struct {
	logger            logging.Logger
	db                db.MoonlightDB
	aws               *aws.Client
	twirpHandler      http.Handler
	daemonHttpClient  *http.Client
	publicMediaURL    string
	jobRunnerQueueURL string
	rtmpSourceURL     string
	ecrRepositoryURL  string
}

func NewControlServer(db db.MoonlightDB, aws *aws.Client, certGen certgen.Generator, jobRunnerQueueURL, rtmpSourceURL, ecrRepositoryURL, publicMediaURL string, logger logging.Logger) (*ControlAPI, error) {
	// We're using acm-ca-go to generate TLS certificates signed by a private CA, so we need to grab the private CA root cert and add it to the pool
	rootCAs, err := ca.GetRootCAs(certGen)
	if err != nil {
		return nil, err
	}

	server := &ControlAPI{
		logger:            logger,
		db:                db,
		aws:               aws,
		jobRunnerQueueURL: jobRunnerQueueURL,
		publicMediaURL:    publicMediaURL,
		rtmpSourceURL:     rtmpSourceURL,
		ecrRepositoryURL:  ecrRepositoryURL,
		daemonHttpClient: &http.Client{
			Transport: &http.Transport{
				TLSClientConfig: &tls.Config{
					RootCAs: rootCAs,
				},
			},
		},
	}
	server.twirpHandler = controlRPC.NewInternalControllerServer(server, nil)

	return server, nil
}

// Handle handles API Gateway Proxy requests and returns the appropriate responses
func (api *ControlAPI) Handle(ctx context.Context, request events.ALBTargetGroupRequest) (events.ALBTargetGroupResponse, error) {
	resp, err := ProcessRequest(ctx, request, binaryContentTypes, api.twirpHandler)
	return resp, err
}

func (api *ControlAPI) SubmitRTMPEvent(ctx context.Context, req *controlRPC.SubmitRTMPEventReq) (*controlRPC.SubmitRTMPEventResp, error) {
	instances, err := api.db.ListInstancesByRtmpKey(req.StreamId)

	if err != nil {
		return nil, err
	}

	if len(instances) == 0 {
		return nil, twirp.NewError(twirp.NotFound, "No RTMP sources were found for the supplied RTMP key")
	}

	if len(instances) != 1 {
		return nil, fmt.Errorf("Expected 1 result, got %v", len(instances))
	}

	instance := instances[0]

	server, err := api.db.GetDaemon(instance.ServerID)

	if err != nil {
		return nil, err
	}

	client := daemonRPC.NewDaemonJSONClient(server.RPCURL, api.daemonHttpClient)

	scene := "Main"

	switch req.Event {
	case types.RTMPEvent_StreamStarvationStart:
	case types.RTMPEvent_StreamBitrateLow:
		scene = "BRB"
	case types.RTMPEvent_StreamStarvationEnd:
	case types.RTMPEvent_StreamBitrateOK:
		scene = "Main"
	case types.RTMPEvent_StreamEnded:
	case types.RTMPEvent_StreamDisconnected:
		scene = "Black"
	}

	resp, err := client.SwitchScene(ctx, &daemonRPC.SwitchSceneReq{
		InstanceId:      instance.ID,
		SceneIdentifier: scene,
	})

	if err != nil {
		return nil, err
	}

	if resp.Success {
		return &controlRPC.SubmitRTMPEventResp{}, nil
	}

	return nil, fmt.Errorf("SwitchScene failed: %v", resp.Message)
}
