package moonlight

import (
	//"code.justin.tv/coleiain/api-gateway-handler"
	"context"
	"time"

	logging "code.justin.tv/event-engineering/golibs/pkg/logging"
	db "code.justin.tv/event-engineering/moonlight-api/pkg/db"
	types "code.justin.tv/event-engineering/moonlight-api/pkg/rpc"
	adminRPC "code.justin.tv/event-engineering/moonlight-api/pkg/rpc/admin"
	controlRPC "code.justin.tv/event-engineering/moonlight-api/pkg/rpc/control"
	"github.com/golang/protobuf/ptypes"
	"github.com/twitchtv/twirp"
)

/*
	Common functions
*/

func mapDaemonFromDb(dbDaemon *db.Daemon, logger logging.Logger) *types.Daemon {
	daemon := &types.Daemon{
		ServerId:     dbDaemon.ID,
		RpcUrl:       dbDaemon.RPCURL,
		Status:       string(dbDaemon.Status),
		NumInstances: dbDaemon.NumInstances,
	}

	lastUpdated, err := ptypes.TimestampProto(dbDaemon.LastUpdated)

	if err == nil {
		daemon.LastUpdated = lastUpdated
	} else {
		logger.Warnf("Failed to serialise protobuf timestamp %v with error %v", dbDaemon.LastUpdated, err)
	}

	return daemon
}

/*
	Control API
*/

func (api *ControlAPI) RegisterDaemon(ctx context.Context, req *controlRPC.RegisterDaemonReq) (*controlRPC.RegisterDaemonResp, error) {
	daemon := &db.Daemon{
		ID:          req.ServerId,
		RPCURL:      req.ApiUrl,
		Status:      db.DaemonReady,
		LastUpdated: time.Now(),
	}
	err := api.db.AddDaemon(daemon)

	if err != nil {
		return nil, err
	}

	resp := &controlRPC.RegisterDaemonResp{
		Success: true,
	}

	return resp, nil
}

func (api *ControlAPI) DeregisterDaemon(ctx context.Context, req *controlRPC.DeregisterDaemonReq) (*controlRPC.DeregisterDaemonResp, error) {
	err := api.db.RemoveDaemon(req.ServerId)

	if err != nil {
		return nil, err
	}

	resp := &controlRPC.DeregisterDaemonResp{
		Success: true,
	}

	return resp, nil
}

func (api *ControlAPI) UpdateDaemonStatus(ctx context.Context, req *controlRPC.UpdateDaemonStatusReq) (*controlRPC.UpdateDaemonStatusResp, error) {
	daemon, err := api.db.GetDaemon(req.ServerId)

	if err != nil {
		return nil, err
	}

	daemon.NumInstances = req.NumInstances
	err = api.db.UpdateDaemon(daemon)

	if err != nil {
		return nil, err
	}

	return &controlRPC.UpdateDaemonStatusResp{
		Success: true,
	}, nil
}

/*
	Admin API
*/

func (api *AdminAPI) GetDaemon(ctx context.Context, req *adminRPC.GetDaemonReq) (*adminRPC.GetDaemonResp, error) {
	canAccessSystem, err := api.hasPermission(ctx, api.bindleLockConfig.CanAccessSystemBindleLockID)

	if err != nil {
		return nil, err
	}

	if !canAccessSystem {
		return nil, twirp.NewError(twirp.PermissionDenied, accessDeniedError)
	}

	daemon, err := api.db.GetDaemon(req.ServerId)

	if err != nil {
		return nil, err
	}

	resp := &adminRPC.GetDaemonResp{
		Daemon: mapDaemonFromDb(daemon, api.logger),
	}

	return resp, nil
}

func (api *AdminAPI) ListDaemons(ctx context.Context, req *adminRPC.ListDaemonsReq) (*adminRPC.ListDaemonsResp, error) {
	canAccessSystem, err := api.hasPermission(ctx, api.bindleLockConfig.CanAccessSystemBindleLockID)

	if err != nil {
		return nil, err
	}

	if !canAccessSystem {
		return nil, twirp.NewError(twirp.PermissionDenied, accessDeniedError)
	}

	daemons, err := api.db.ListDaemons()

	if err != nil {
		return nil, err
	}

	resp := &adminRPC.ListDaemonsResp{
		Daemons: make([]*types.Daemon, 0),
	}

	for _, dbDaemon := range daemons {
		resp.Daemons = append(resp.Daemons, mapDaemonFromDb(dbDaemon, api.logger))
	}

	return resp, nil
}
