package moonlight

import (
	"context"
	"errors"
	"fmt"
	"net/http"
	"strings"
	"time"

	logging "code.justin.tv/event-engineering/golibs/pkg/logging"
	db "code.justin.tv/event-engineering/moonlight-api/pkg/db"
	jobrunner "code.justin.tv/event-engineering/moonlight-api/pkg/jobrunner"
	types "code.justin.tv/event-engineering/moonlight-api/pkg/rpc"
	adminRPC "code.justin.tv/event-engineering/moonlight-api/pkg/rpc/admin"
	controlRPC "code.justin.tv/event-engineering/moonlight-api/pkg/rpc/control"
	obs "code.justin.tv/event-engineering/moonlight-daemon/pkg/obs"
	daemonRPC "code.justin.tv/event-engineering/moonlight-daemon/pkg/rpc"
	"github.com/golang/protobuf/ptypes"
	"github.com/google/uuid"
	"github.com/twitchtv/twirp"
)

/*
	Common functions
*/

func stopInstance(instanceID string, db db.MoonlightDB, daemonHttpClient *http.Client, logger logging.Logger) (bool, error) {
	instance, err := db.GetInstance(instanceID)

	if err != nil {
		return false, err
	}

	if instance.ServerID != "" && instance.ServerID != "unassigned" {
		server, err := db.GetDaemon(instance.ServerID)
		if err != nil {
			logger.Warnf("Error getting server, %v", err)
			return false, err
		}

		client := daemonRPC.NewDaemonJSONClient(server.RPCURL, daemonHttpClient)

		daemonContext, cancel := context.WithTimeout(context.Background(), 30*time.Second)
		defer cancel()

		output, err := client.StopInstance(daemonContext, &daemonRPC.StopInstanceReq{
			InstanceId: instance.ID,
		})

		if err != nil {
			return false, err
		}

		// TODO if the instance is not found then we should proceed to remove from the DB
		// The underluying function doesn't explicitly say when it's not found yet, TODOOOO
		if !output.Success {
			return false, fmt.Errorf("Failed to stop Instance %v", output.Message)
		}
	}

	err = db.RemoveInstance(instance.ID)

	if err != nil {
		return false, err
	}

	return true, nil
}

func mapInstanceFromDb(dbInstance *db.Instance, logger logging.Logger) *types.Instance {
	instance := &types.Instance{
		InstanceId:      dbInstance.ID,
		ServerId:        dbInstance.ServerID,
		TwitchUserId:    dbInstance.TwitchUserID,
		RtmpKey:         dbInstance.RTMPKey,
		ObsScenesJson:   dbInstance.OBSScenesJSON,
		DockerImageName: dbInstance.DockerImageName,
		HeartbeatJson:   dbInstance.Heartbeat,
		Status:          string(dbInstance.Status),
	}

	lastUpdated, err := ptypes.TimestampProto(dbInstance.LastUpdated)

	if err == nil {
		instance.LastUpdated = lastUpdated
	} else {
		logger.Warnf("Failed to serialise last updated protobuf timestamp %v with error %v", dbInstance.LastUpdated, err)
	}

	lastStreamed, err := ptypes.TimestampProto(dbInstance.LastStreamed)
	if err == nil {
		instance.LastStreamed = lastStreamed
	} else {
		logger.Warnf("Failed to serialise last streamed protobuf timestamp %v with error %v", dbInstance.LastStreamed, err)
	}

	return instance
}

/*
	Control API
*/

func (api *ControlAPI) InitialiseSession(ctx context.Context, req *controlRPC.InitialiseSessionReq) (*controlRPC.InitialiseSessionResp, error) {
	user, err := api.db.GetUser(req.GetTwitchUserId())

	if err != nil {
		api.logger.Warnf("Error getting user, %v", err)
		return nil, twirp.NotFoundError("Could not find user")
	}

	if user.RTMPSource == "" {
		// We need the user to tell us which RTMP source they have, TODO: work out how to return the right thing here
		api.logger.Warn("Unable to determine RTMP source")
		return nil, errors.New("Unable to determine RTMP source, please update user settings")
	}

	var instance *db.Instance

	// If there's an instance already there then return it
	instances, err := api.db.ListInstancesByTwitchUserID(user.TwitchUserID)
	if err == nil && instances != nil && len(instances) > 0 {
		instance = instances[0]
	} else {
		rtmpSource, srcErr := api.db.GetRTMPSource(user.RTMPSource)

		if srcErr != nil {
			api.logger.Warnf("Error getting RTMP source, %v", srcErr)
			return nil, srcErr
		}

		// Create new instance
		instance = &db.Instance{
			ID:              uuid.New().String(),
			TwitchUserID:    req.GetTwitchUserId(),
			OBSScenesJSON:   user.OBSScenesJSON,
			RTMPKey:         rtmpSource.RTMPKey,
			Status:          db.InstanceNew,
			LastUpdated:     time.Now(),
			LastStreamed:    time.Now(),
			ServerID:        "unassigned",
			DockerImageName: fmt.Sprintf("%s:latest", api.ecrRepositoryURL),
		}

		err = api.db.AddInstance(instance)

		if err != nil {
			api.logger.Warnf("Error creating instance, %v", err)
			return nil, err
		}

		message := jobrunner.AllocateServerMessage{
			InstanceID: instance.ID,
		}

		err = api.aws.SendSQSJob(api.jobRunnerQueueURL, jobrunner.AllocateServer.String(), 0, message)

		if err != nil {
			api.logger.Warnf("Error sending SQS job, %v", err)
			return nil, err
		}
	}

	scenes, err := obs.ParseScenes(instance.OBSScenesJSON)
	if err != nil {
		api.logger.Warnf("Unable to parse scenes, %v", err)
		return nil, errors.New("Unable to parse scenes")
	}

	return &controlRPC.InitialiseSessionResp{
		InstanceId: instance.ID,
		InstanceInfo: &controlRPC.GetInstanceStatusResp{
			Status:        string(instance.Status),
			HeartbeatJson: instance.Heartbeat,
		},
		Scenes: scenes,
	}, nil
}

func (api *ControlAPI) GetInstanceStatus(ctx context.Context, req *controlRPC.GetInstanceStatusReq) (*controlRPC.GetInstanceStatusResp, error) {
	var instance *db.Instance

	// If there's an instance already there then return it
	instances, err := api.db.ListInstancesByTwitchUserID(req.TwitchUserId)
	if err == nil && instances != nil && len(instances) > 0 {
		instance = instances[0]
		return &controlRPC.GetInstanceStatusResp{
			Status:        string(instance.Status),
			RtmpMuted:     instance.RTMPMuted,
			HeartbeatJson: instance.Heartbeat,
		}, nil
	}

	return nil, twirp.NewError(twirp.NotFound, "No instance found")
}

func (api *ControlAPI) SetInstanceStatus(ctx context.Context, req *controlRPC.SetInstanceStatusReq) (*controlRPC.SetInstanceStatusResp, error) {
	// For now we're just going to do this, but we need to do this with a proper update
	instance, err := api.db.GetInstance(req.GetInstanceId())
	if err != nil {
		return nil, err
	}

	instance.Status = db.InstanceStatus(req.GetStatus())
	instance.RTMPMuted = req.RtmpMuted
	instance.Heartbeat = req.HeartbeatJson
	instance.LastUpdated = time.Now()

	lastStreamed, err := ptypes.Timestamp(req.LastStreamed)
	if err != nil {
		api.logger.Warnf("Failed to convert protobuf timestamp to Time %v", err)
	} else if !time.Time.IsZero(lastStreamed) {
		instance.LastStreamed = lastStreamed
	}

	err = api.db.UpdateInstance(instance)
	if err != nil {
		return nil, err
	}

	return &controlRPC.SetInstanceStatusResp{}, nil
}

func (api *ControlAPI) BatchSetInstanceStatus(ctx context.Context, req *controlRPC.BatchSetInstanceStatusReq) (*controlRPC.BatchSetInstanceStatusResp, error) {
	errors := make([]string, 0)

	for _, update := range req.Statuses {
		_, err := api.SetInstanceStatus(ctx, update)
		if err != nil {
			errors = append(errors, fmt.Sprintf("Failed to update instance %v to status %v with error %v", update.InstanceId, update.Status, err.Error()))
		}
	}

	if len(errors) > 0 {
		return &controlRPC.BatchSetInstanceStatusResp{}, fmt.Errorf("Batch update had at least 1 failure: %v", strings.Join(errors, ", "))
	}

	return &controlRPC.BatchSetInstanceStatusResp{}, nil
}

// This exists on both the admin and control APIs
func (api *ControlAPI) StopInstance(ctx context.Context, req *controlRPC.StopInstanceReq) (*controlRPC.StopInstanceResp, error) {
	success, err := stopInstance(req.GetInstanceId(), api.db, api.daemonHttpClient, api.logger)

	if err != nil {
		return nil, err
	}

	resp := &controlRPC.StopInstanceResp{
		Success: success,
	}

	return resp, nil
}

func (api *ControlAPI) GetOBSCloudConfig(ctx context.Context, req *controlRPC.GetOBSCloudConfigReq) (*controlRPC.GetOBSCloudConfigResp, error) {
	api.logger.Debug("GetOBSCloudConfig: start")
	instances, err := api.db.ListInstancesByServer(req.ServerId)

	if err != nil {
		return nil, err
	}

	api.logger.Debugf("GetOBSCloudConfig: got %v instances", len(instances))

	config := &controlRPC.GetOBSCloudConfigResp{
		Instances: make([]*types.OBSCloudInstance, 0),
	}

	for _, instance := range instances {
		config.Instances = append(config.Instances, &types.OBSCloudInstance{
			Id:              instance.ID,
			ServerId:        instance.ServerID,
			DockerImageName: instance.DockerImageName,
			ObsConfig: &types.OBSConfig{
				ScenesJson:          instance.OBSScenesJSON,
				RtmpSourceUrl:       fmt.Sprintf("rtmp://%s/app/%s", api.rtmpSourceURL, instance.RTMPKey),
				AudioDeviceId:       "pulse_output_capture", // Not really sure where this should go, it's the same for all instances
				AudioDeviceDeviceId: "MySink.monitor",
			},
		})
	}

	api.logger.Debug("GetOBSCloudConfig: end")

	return config, nil
}

/*
	Admin API
*/

func (api *AdminAPI) GetInstanceAccessKey(ctx context.Context, req *adminRPC.GetInstanceAccessKeyReq) (*adminRPC.GetInstanceAccessKeyResp, error) {
	hasPermission, err := api.hasPermission(ctx, api.bindleLockConfig.AdminBindleLockID)

	if err != nil {
		return nil, err
	}

	if !hasPermission {
		return nil, twirp.NewError(twirp.PermissionDenied, accessDeniedError)
	}

	instance, err := api.db.GetInstance(req.GetInstanceId())

	if err != nil {
		return nil, err
	}

	server, err := api.db.GetDaemon(instance.ServerID)
	if err != nil {
		api.logger.Warnf("Error getting server, %v", err)
		return nil, err
	}

	client := daemonRPC.NewDaemonJSONClient(server.RPCURL, api.daemonHttpClient)

	vncInfo, err := client.GetVNCKey(ctx, &daemonRPC.GetVNCKeyReq{
		InstanceId: instance.ID,
	})

	if err != nil {
		return nil, err
	}

	return &adminRPC.GetInstanceAccessKeyResp{
		AwsInstanceId: instance.ServerID,
		AccessKey:     vncInfo.VncKey,
		Port:          vncInfo.VncPort,
	}, nil
}

// This exists on both the admin and control APIs
func (api *AdminAPI) StopInstance(ctx context.Context, req *adminRPC.StopInstanceReq) (*adminRPC.StopInstanceResp, error) {
	hasPermission, err := api.hasPermission(ctx, api.bindleLockConfig.OpsBindleLockID)

	if err != nil {
		return nil, err
	}

	if !hasPermission {
		return nil, twirp.NewError(twirp.PermissionDenied, accessDeniedError)
	}

	success, err := stopInstance(req.GetInstanceId(), api.db, api.daemonHttpClient, api.logger)

	if err != nil {
		return nil, err
	}

	resp := &adminRPC.StopInstanceResp{
		Success: success,
	}

	return resp, nil
}

func (api *AdminAPI) GetInstance(ctx context.Context, req *adminRPC.GetInstanceReq) (*adminRPC.GetInstanceResp, error) {
	hasPermission, err := api.hasPermission(ctx, api.bindleLockConfig.CanAccessSystemBindleLockID)

	if err != nil {
		return nil, err
	}

	if !hasPermission {
		return nil, twirp.NewError(twirp.PermissionDenied, accessDeniedError)
	}

	instance, err := api.db.GetInstance(req.InstanceId)

	if err != nil {
		return nil, err
	}

	resp := &adminRPC.GetInstanceResp{
		Instance: mapInstanceFromDb(instance, api.logger),
	}

	return resp, nil
}

func (api *AdminAPI) ListInstances(ctx context.Context, req *adminRPC.ListInstancesReq) (*adminRPC.ListInstancesResp, error) {
	hasPermission, err := api.hasPermission(ctx, api.bindleLockConfig.CanAccessSystemBindleLockID)

	if err != nil {
		return nil, err
	}

	if !hasPermission {
		return nil, twirp.NewError(twirp.PermissionDenied, accessDeniedError)
	}

	instances, err := api.db.ListInstances()

	if err != nil {
		return nil, err
	}

	resp := &adminRPC.ListInstancesResp{
		Instances: make([]*types.Instance, 0),
	}

	for _, dbInstance := range instances {
		resp.Instances = append(resp.Instances, mapInstanceFromDb(dbInstance, api.logger))
	}

	return resp, nil
}
