package moonlight

import (
	"context"

	db "code.justin.tv/event-engineering/moonlight-api/pkg/db"
	types "code.justin.tv/event-engineering/moonlight-api/pkg/rpc"
	adminRPC "code.justin.tv/event-engineering/moonlight-api/pkg/rpc/admin"
	controlRPC "code.justin.tv/event-engineering/moonlight-api/pkg/rpc/control"
	"github.com/google/uuid"
	"github.com/twitchtv/twirp"
)

/*
	Common functions
*/

func mapRtmpSourceFromDb(dbSource *db.RTMPSource) *types.RTMPSource {
	return &types.RTMPSource{
		SourceId:     dbSource.SourceID,
		RtmpKey:      dbSource.RTMPKey,
		InternalAddr: dbSource.InternalAddr,
	}
}

func mapRtmpSourceFromRpc(rpcSource *types.RTMPSource) *db.RTMPSource {
	return &db.RTMPSource{
		SourceID:     rpcSource.SourceId,
		RTMPKey:      rpcSource.RtmpKey,
		InternalAddr: rpcSource.InternalAddr,
	}
}

/*
	Control API
*/

func (api *ControlAPI) RegisterStream(ctx context.Context, req *controlRPC.RegisterStreamReq) (*controlRPC.RegisterStreamResp, error) {
	rtmpSource, err := api.db.GetRTMPSourceByRTMPKey(req.GetRtmpKey())

	if err != nil {
		return nil, err
	}

	rtmpSource.InternalAddr = req.GetListenAddr()

	err = api.db.UpdateRTMPSource(rtmpSource)

	if err != nil {
		return nil, err
	}

	resp := &controlRPC.RegisterStreamResp{
		Success: true,
	}

	return resp, nil
}

func (api *ControlAPI) GetRTMPSourceByRTMPKey(ctx context.Context, req *controlRPC.GetRTMPSourceByRTMPKeyReq) (*controlRPC.GetRTMPSourceByRTMPKeyResp, error) {
	rtmpSource, err := api.db.GetRTMPSourceByRTMPKey(req.GetRtmpKey())

	if err != nil {
		return nil, err
	}

	ret := &types.RTMPSource{
		SourceId:     rtmpSource.SourceID,
		RtmpKey:      rtmpSource.RTMPKey,
		InternalAddr: rtmpSource.InternalAddr,
	}

	resp := &controlRPC.GetRTMPSourceByRTMPKeyResp{
		Success:    true,
		RtmpSource: ret,
	}

	return resp, nil
}

/*
	Admin API
*/

func (api *AdminAPI) CreateRTMPSource(ctx context.Context, req *adminRPC.CreateRTMPSourceReq) (*adminRPC.CreateRTMPSourceResp, error) {
	hasPermission, err := api.hasPermission(ctx, api.bindleLockConfig.OpsBindleLockID)

	if err != nil {
		return nil, err
	}

	if !hasPermission {
		return nil, twirp.NewError(twirp.PermissionDenied, accessDeniedError)
	}

	rtmpKey := uuid.New().String()

	source := &db.RTMPSource{
		SourceID: req.GetSourceId(),
		RTMPKey:  rtmpKey,
	}
	err = api.db.AddRTMPSource(source)

	if err != nil {
		return nil, err
	}

	resp := &adminRPC.CreateRTMPSourceResp{
		Success: true,
	}

	return resp, nil
}

func (api *AdminAPI) GetRTMPSource(ctx context.Context, req *adminRPC.GetRTMPSourceReq) (*adminRPC.GetRTMPSourceResp, error) {
	hasPermission, err := api.hasPermission(ctx, api.bindleLockConfig.CanAccessSystemBindleLockID)

	if err != nil {
		return nil, err
	}

	if !hasPermission {
		return nil, twirp.NewError(twirp.PermissionDenied, accessDeniedError)
	}

	source, err := api.db.GetRTMPSource(req.SourceId)

	if err != nil {
		return nil, err
	}

	resp := &adminRPC.GetRTMPSourceResp{
		RtmpSource: mapRtmpSourceFromDb(source),
	}

	return resp, nil
}

func (api *AdminAPI) ListRTMPSources(ctx context.Context, req *adminRPC.ListRTMPSourcesReq) (*adminRPC.ListRTMPSourcesResp, error) {
	hasPermission, err := api.hasPermission(ctx, api.bindleLockConfig.CanAccessSystemBindleLockID)

	if err != nil {
		return nil, err
	}

	if !hasPermission {
		return nil, twirp.NewError(twirp.PermissionDenied, accessDeniedError)
	}

	rtmpSources, err := api.db.ListRTMPSources()

	if err != nil {
		return nil, err
	}

	resp := &adminRPC.ListRTMPSourcesResp{
		RtmpSources: make([]*types.RTMPSource, 0),
	}

	for _, dbSource := range rtmpSources {
		resp.RtmpSources = append(resp.RtmpSources, mapRtmpSourceFromDb(dbSource))
	}

	return resp, nil
}

func (api *AdminAPI) RemoveRTMPSource(ctx context.Context, req *adminRPC.RemoveRTMPSourceReq) (*adminRPC.RemoveRTMPSourceResp, error) {
	hasPermission, err := api.hasPermission(ctx, api.bindleLockConfig.OpsBindleLockID)

	if err != nil {
		return nil, err
	}

	if !hasPermission {
		return nil, twirp.NewError(twirp.PermissionDenied, accessDeniedError)
	}

	err = api.db.RemoveRTMPSource(req.SourceId)

	if err != nil {
		return nil, err
	}

	resp := &adminRPC.RemoveRTMPSourceResp{}

	return resp, nil
}

func (api *AdminAPI) UpdateRTMPSource(ctx context.Context, req *adminRPC.UpdateRTMPSourceReq) (*adminRPC.UpdateRTMPSourceResp, error) {
	hasPermission, err := api.hasPermission(ctx, api.bindleLockConfig.OpsBindleLockID)

	if err != nil {
		return nil, err
	}

	if !hasPermission {
		return nil, twirp.NewError(twirp.PermissionDenied, accessDeniedError)
	}

	_, err = api.db.GetRTMPSource(req.RtmpSource.SourceId)

	if err != nil {
		return nil, err
	}

	updatedSource := mapRtmpSourceFromRpc(req.RtmpSource)

	err = api.db.UpdateRTMPSource(updatedSource)

	if err != nil {
		return nil, err
	}

	source, err := api.db.GetRTMPSource(req.RtmpSource.SourceId)
	if err != nil {
		return nil, err
	}

	resp := &adminRPC.UpdateRTMPSourceResp{
		RtmpSource: mapRtmpSourceFromDb(source),
	}

	return resp, nil
}
