package moonlight

import (
	db "code.justin.tv/event-engineering/moonlight-api/pkg/db"
	controlRPC "code.justin.tv/event-engineering/moonlight-api/pkg/rpc/control"
	daemonRPC "code.justin.tv/event-engineering/moonlight-daemon/pkg/rpc"
	"context"
	"errors"
)

/*
	Common functions
*/

func (api *ControlAPI) getInstance(twitchUserID string) (*db.Instance, error) {
	instances, err := api.db.ListInstancesByTwitchUserID(twitchUserID)
	if err != nil {
		return nil, err
	}

	if len(instances) == 0 {
		return nil, errors.New("No instances found")
	}

	if len(instances) != 1 {
		return nil, errors.New("Multiple instances found")
	}

	return instances[0], nil
}

/*
	Control API
*/

func (api *ControlAPI) SetStreamLive(ctx context.Context, req *controlRPC.SetStreamLiveReq) (*controlRPC.StreamOperationResp, error) {
	if req.GetTwitchUserId() == "" {
		api.logger.Warn("No Twitch User ID passed")
		return nil, errors.New("No Twitch User ID passed")
	}

	instance, err := api.getInstance(req.GetTwitchUserId())
	if err != nil {
		api.logger.Warnf("Error getting instance, %v", err)
		return nil, err
	}

	server, err := api.db.GetDaemon(instance.ServerID)
	if err != nil {
		api.logger.Warnf("Error getting server, %v", err)
		return nil, err
	}

	client := daemonRPC.NewDaemonJSONClient(server.RPCURL, api.daemonHttpClient)

	if req.GetLive() {
		resp, err := client.StartStream(ctx, &daemonRPC.StartStreamReq{
			InstanceId: instance.ID,
			StreamKey:  req.GetStreamKey() + "?broadcaster=moonlight",
		})

		if err != nil {
			api.logger.Warnf("Error starting stream, %v", err)
			return nil, err
		}

		if !resp.GetSuccess() {
			api.logger.Warnf("Error starting stream, %v", resp.GetMessage())
			return nil, errors.New(resp.GetMessage())
		}
	} else {
		resp, err := client.StopStream(ctx, &daemonRPC.StopStreamReq{
			InstanceId: instance.ID,
		})

		if err != nil {
			api.logger.Warnf("Error stopping stream, %v", err)
			return nil, err
		}

		if !resp.GetSuccess() {
			api.logger.Warnf("Error stopping stream, %v", resp.GetMessage())
			return nil, errors.New(resp.GetMessage())
		}
	}

	return &controlRPC.StreamOperationResp{
		Success: true,
	}, nil
}

func (api *ControlAPI) SetStreamMute(ctx context.Context, req *controlRPC.SetStreamMuteReq) (*controlRPC.StreamOperationResp, error) {
	if req.GetTwitchUserId() == "" {
		return nil, errors.New("No Twitch User ID passed")
	}

	instance, err := api.getInstance(req.GetTwitchUserId())
	if err != nil {
		return nil, err
	}

	server, err := api.db.GetDaemon(instance.ServerID)
	if err != nil {
		return nil, err
	}

	client := daemonRPC.NewDaemonJSONClient(server.RPCURL, api.daemonHttpClient)

	resp, err := client.MuteInput(ctx, &daemonRPC.MuteInputReq{
		InstanceId: instance.ID,
		Mute:       req.GetMute(),
	})

	if err != nil {
		return nil, err
	}

	if !resp.Success {
		return &controlRPC.StreamOperationResp{
			Success: false,
			Message: resp.Message,
		}, nil
	}

	return &controlRPC.StreamOperationResp{
		Success: true,
	}, nil
}

func (api *ControlAPI) SetStreamScene(ctx context.Context, req *controlRPC.SetStreamSceneReq) (*controlRPC.StreamOperationResp, error) {
	if req.GetTwitchUserId() == "" {
		return nil, errors.New("No Twitch User ID passed")
	}

	if req.GetSceneIdentifier() == "" {
		return nil, errors.New("No Scene Identifier passed")
	}

	instance, err := api.getInstance(req.GetTwitchUserId())
	if err != nil {
		return nil, err
	}

	server, err := api.db.GetDaemon(instance.ServerID)
	if err != nil {
		return nil, err
	}

	client := daemonRPC.NewDaemonJSONClient(server.RPCURL, api.daemonHttpClient)

	resp, err := client.SwitchScene(ctx, &daemonRPC.SwitchSceneReq{
		InstanceId:      instance.ID,
		SceneIdentifier: req.GetSceneIdentifier(),
	})

	if err != nil {
		api.logger.Warnf("Error calling SwitchScene %v", err)
		return nil, err
	}

	if !resp.GetSuccess() {
		err = errors.New(resp.GetMessage())
		api.logger.Warnf("SwitchScene failed %v", err)
		return nil, err
	}

	return &controlRPC.StreamOperationResp{
		Success: true,
	}, nil
}
