package moonlight

import (
	"context"
	"fmt"
	"net/url"
	"regexp"
	"strconv"

	db "code.justin.tv/event-engineering/moonlight-api/pkg/db"
	"code.justin.tv/event-engineering/moonlight-api/pkg/obs"
	types "code.justin.tv/event-engineering/moonlight-api/pkg/rpc"
	adminRPC "code.justin.tv/event-engineering/moonlight-api/pkg/rpc/admin"
	controlRPC "code.justin.tv/event-engineering/moonlight-api/pkg/rpc/control"
	daemonRPC "code.justin.tv/event-engineering/moonlight-daemon/pkg/rpc"
	"github.com/twitchtv/twirp"
)

var urlRegex = regexp.MustCompile("^https://.*$")

/*
	Common functions
*/

func mapUserFromDb(dbUser *db.User) *types.User {
	return &types.User{
		TwitchUserId:  dbUser.TwitchUserID,
		Active:        dbUser.Active,
		RtmpSource:    dbUser.RTMPSource,
		ObsScenesJson: dbUser.OBSScenesJSON,
	}
}

func mapUserFromRpc(rpcUser *types.User) *db.User {
	return &db.User{
		TwitchUserID:  rpcUser.TwitchUserId,
		Active:        rpcUser.Active,
		RTMPSource:    rpcUser.RtmpSource,
		OBSScenesJSON: rpcUser.ObsScenesJson,
	}
}

/*
	Control API
*/

func (api *ControlAPI) UpdateOBSConfig(ctx context.Context, req *controlRPC.UpdateOBSConfigReq) (*controlRPC.UpdateOBSConfigResp, error) {
	// This will error if the user does not exist
	user, err := api.db.GetUser(req.TwitchUserId)

	if err != nil {
		return nil, err
	}

	if req.MainOverlayUrl != "" {
		parsedUrl, err := url.Parse(req.MainOverlayUrl)
		if err != nil {
			return &controlRPC.UpdateOBSConfigResp{
				Success: false,
				Message: "Failed to parse URL",
			}, nil
		}

		if parsedUrl.Scheme != "https" {
			return &controlRPC.UpdateOBSConfigResp{
				Success: false,
				Message: "Please supply a valid https URL",
			}, nil
		}
	}

	// Try to generate OBS scenes if applicable
	//if req.MainOverlayUrl != "" {
	scenesJSON, err := obs.GenerateOBSScenes(obs.OBSTemplateRepl{
		BRBBrowserSource:         fmt.Sprintf("%v/moonlight_brb.png", api.publicMediaURL),
		MainOverlayBrowserSource: req.MainOverlayUrl,
		RtmpSourceUrl:            "{{.RtmpSourceUrl}}", // We actually want to leave these 3 in to be replaced at a later step but template will complain if we dont replace it with something
		AudioDeviceId:            "{{.AudioDeviceId}}", // the case is different on these properties because they're currently populated by something generated by Twirp
		AudioDeviceDeviceId:      "{{.AudioDeviceDeviceId}}",
		MainOverlayWidth:         strconv.Itoa(int(req.MainOverlayWidth)),
		MainOverlayHeight:        strconv.Itoa(int(req.MainOverlayHeight)),
		MainOverlayX:             strconv.Itoa(int(req.MainOverlayX)),
		MainOverlayY:             strconv.Itoa(int(req.MainOverlayY)),
	})

	if err != nil {
		return &controlRPC.UpdateOBSConfigResp{
			Success: false,
			Message: "Failed to generate OBS config: " + err.Error(),
		}, nil
	}

	// Update the user
	user.OBSScenesJSON = scenesJSON
	err = api.db.UpdateUser(user)
	if err != nil {
		return &controlRPC.UpdateOBSConfigResp{
			Success: false,
			Message: "Failed to save OBS config: " + err.Error(),
		}, nil
	}

	// Update the source directly on any running instances
	instances, err := api.db.ListInstancesByTwitchUserID(user.TwitchUserID)
	if err != nil {
		return nil, err
	}

	for _, instance := range instances {
		server, err := api.db.GetDaemon(instance.ServerID)
		if err != nil {
			api.logger.Warnf("Error getting server, %v", err)
			return nil, err
		}

		client := daemonRPC.NewDaemonJSONClient(server.RPCURL, api.daemonHttpClient)

		resp, err := client.UpdateOBSConfig(context.Background(), &daemonRPC.UpdateOBSConfigReq{
			InstanceId:        instance.ID,
			MainOverlayUrl:    req.MainOverlayUrl,
			MainOverlayWidth:  req.MainOverlayWidth,
			MainOverlayHeight: req.MainOverlayHeight,
			MainOverlayX:      req.MainOverlayX,
			MainOverlayY:      req.MainOverlayY,
		})

		if err != nil {
			api.logger.Warnf("Error updating OBS config, %v", err)
			return nil, err
		}

		if !resp.Success {
			api.logger.Warnf("Error updating OBS config, %v", resp.Message)
			return &controlRPC.UpdateOBSConfigResp{
				Success: false,
				Message: resp.Message,
			}, nil
		}
	}

	return &controlRPC.UpdateOBSConfigResp{
		Success: true,
	}, nil
}

/*
	Admin API
*/

func (api *AdminAPI) removeSourceFromUsers(users []string) error {
	// Remove the source from the other users and kill any instances that they have running
	for _, tuid := range users {
		user, err := api.db.GetUser(tuid)
		if err != nil {
			return err
		}

		// Remove source from user
		user.RTMPSource = ""
		err = api.db.UpdateUser(user)
		if err != nil {
			return err
		}

		instances, err := api.db.ListInstancesByTwitchUserID(tuid)
		if err != nil {
			return err
		}

		// This could be really slow
		for _, instance := range instances {
			_, err := api.StopInstance(context.Background(), &adminRPC.StopInstanceReq{
				InstanceId: instance.ID,
			})
			if err != nil {
				return err
			}
		}
	}

	return nil
}

func (api *AdminAPI) CreateUser(ctx context.Context, req *adminRPC.CreateUserReq) (*adminRPC.CreateUserResp, error) {
	hasPermission, err := api.hasPermission(ctx, api.bindleLockConfig.OpsBindleLockID)

	if err != nil {
		return nil, err
	}

	if !hasPermission {
		return nil, twirp.NewError(twirp.PermissionDenied, accessDeniedError)
	}

	// Check to see if the RTMP source is in use
	user := &db.User{
		TwitchUserID: req.GetTwitchUserId(),
		RTMPSource:   req.GetRtmpSource(),
		Active:       true,
	}

	usersWithSource, err := api.anyOtherUsersWithSource(user)

	if err != nil {
		return nil, err
	}

	if len(usersWithSource) > 0 {
		if req.ForceCreate {
			err = api.removeSourceFromUsers(usersWithSource)
			if err != nil {
				return nil, err
			}
		} else {
			return &adminRPC.CreateUserResp{
				Success:                  false,
				Message:                  "There were other users with this source ID",
				ConflictingTwitchUserIds: usersWithSource,
			}, nil
		}
	}

	// Create the user record
	err = api.db.AddUser(user)

	if err != nil {
		return nil, err
	}

	// Try to generate OBS scenes if applicable
	//if req.MainOverlayUrl != "" {
	scenesJSON, err := obs.GenerateOBSScenes(obs.OBSTemplateRepl{
		BRBBrowserSource:         fmt.Sprintf("%v/moonlight_brb.png", api.publicMediaURL),
		MainOverlayBrowserSource: req.MainOverlayUrl,
		RtmpSourceUrl:            "{{.RtmpSourceUrl}}", // We actually want to leave these 3 in to be replaced at a later step but template will complain if we dont replace it with something
		AudioDeviceId:            "{{.AudioDeviceId}}", // the case is different on these properties because they're currently populated by something generated by Twirp
		AudioDeviceDeviceId:      "{{.AudioDeviceDeviceId}}",
		MainOverlayWidth:         strconv.Itoa(int(req.MainOverlayWidth)),
		MainOverlayHeight:        strconv.Itoa(int(req.MainOverlayHeight)),
		MainOverlayX:             strconv.Itoa(int(req.MainOverlayX)),
		MainOverlayY:             strconv.Itoa(int(req.MainOverlayY)),
	})

	if err != nil {
		return &adminRPC.CreateUserResp{
			Success: false,
			Message: "Failed to generate OBS config: " + err.Error(),
		}, nil
	}

	// Update the user
	user.OBSScenesJSON = scenesJSON
	err = api.db.UpdateUser(user)
	if err != nil {
		return &adminRPC.CreateUserResp{
			Success: false,
			Message: "Failed to save OBS config: " + err.Error(),
		}, nil
	}

	resp := &adminRPC.CreateUserResp{
		Success: true,
	}

	return resp, nil
}

// Check the status of the source that we want to assign to this user
func (api *AdminAPI) anyOtherUsersWithSource(dbUser *db.User) ([]string, error) {
	users, err := api.db.ListUsers()
	if err != nil {
		return nil, err
	}

	usersWithSource := make([]string, 0)

	for _, user := range users {
		if user.RTMPSource == dbUser.RTMPSource && user.TwitchUserID != dbUser.TwitchUserID {
			usersWithSource = append(usersWithSource, user.TwitchUserID)
		}
	}

	return usersWithSource, nil
}

func (api *AdminAPI) GetUser(ctx context.Context, req *adminRPC.GetUserReq) (*adminRPC.GetUserResp, error) {
	hasPermission, err := api.hasPermission(ctx, api.bindleLockConfig.CanAccessSystemBindleLockID)

	if err != nil {
		return nil, err
	}

	if !hasPermission {
		return nil, twirp.NewError(twirp.PermissionDenied, accessDeniedError)
	}

	user, err := api.db.GetUser(req.TwitchUserId)

	if err != nil {
		return nil, err
	}

	resp := &adminRPC.GetUserResp{
		User: mapUserFromDb(user),
	}

	return resp, nil
}

func (api *AdminAPI) ListUsers(ctx context.Context, req *adminRPC.ListUsersReq) (*adminRPC.ListUsersResp, error) {
	hasPermission, err := api.hasPermission(ctx, api.bindleLockConfig.CanAccessSystemBindleLockID)

	if err != nil {
		return nil, err
	}

	if !hasPermission {
		return nil, twirp.NewError(twirp.PermissionDenied, accessDeniedError)
	}

	users, err := api.db.ListUsers()

	if err != nil {
		return nil, err
	}

	resp := &adminRPC.ListUsersResp{
		Users: make([]*types.User, 0),
	}

	for _, dbUser := range users {
		resp.Users = append(resp.Users, mapUserFromDb(dbUser))
	}

	return resp, nil
}

func (api *AdminAPI) RemoveUser(ctx context.Context, req *adminRPC.RemoveUserReq) (*adminRPC.RemoveUserResp, error) {
	hasPermission, err := api.hasPermission(ctx, api.bindleLockConfig.OpsBindleLockID)

	if err != nil {
		return nil, err
	}

	if !hasPermission {
		return nil, twirp.NewError(twirp.PermissionDenied, accessDeniedError)
	}

	err = api.db.RemoveUser(req.TwitchUserId)

	if err != nil {
		return nil, err
	}

	resp := &adminRPC.RemoveUserResp{}

	return resp, nil
}

func (api *AdminAPI) UpdateUser(ctx context.Context, req *adminRPC.UpdateUserReq) (*adminRPC.UpdateUserResp, error) {
	hasPermission, err := api.hasPermission(ctx, api.bindleLockConfig.OpsBindleLockID)

	if err != nil {
		return nil, err
	}

	if !hasPermission {
		return nil, twirp.NewError(twirp.PermissionDenied, accessDeniedError)
	}

	// This will error if the user does not exist
	user, err := api.db.GetUser(req.User.TwitchUserId)

	if err != nil {
		return nil, err
	}

	// Currently this is the only thing you can update via this method, we'll need to work out the right way to do this properly
	user.RTMPSource = req.User.RtmpSource

	usersWithSource, err := api.anyOtherUsersWithSource(user)
	if err != nil {
		return nil, err
	}

	if len(usersWithSource) > 0 {
		if req.ForceUpdate {
			err = api.removeSourceFromUsers(usersWithSource)
			if err != nil {
				return nil, err
			}
		} else {
			return &adminRPC.UpdateUserResp{
				Success:                  false,
				Message:                  "There were other users with this source ID",
				ConflictingTwitchUserIds: usersWithSource,
			}, nil
		}
	}

	err = api.db.UpdateUser(user)

	if err != nil {
		return nil, err
	}

	user, err = api.db.GetUser(req.User.TwitchUserId)
	if err != nil {
		return nil, err
	}

	resp := &adminRPC.UpdateUserResp{
		User:    mapUserFromDb(user),
		Success: true,
	}

	return resp, nil
}
