package backend

import (
	logging "code.justin.tv/event-engineering/golibs/pkg/logging"
	"encoding/json"
	"errors"
	"fmt"
	"net/http"
)

type ValidateTokenResponse struct {
	ClientID string   `json:"client_id"`
	Login    string   `json:"login"`
	Scopes   []string `json:"scopes"`
	UserID   string   `json:"user_id"`
}

type GetChannelResponse struct {
	ID        string `json:"_id"`
	StreamKey string `json:"stream_key"`
}

// Client is an interface containing methods that interact with Twitch
type Client interface {
	ValidateToken(token string) (ValidateTokenResponse, error)
	GetChannel(token string) (GetChannelResponse, error)
	GetClientID() string
}

type client struct {
	logger     logging.Logger
	clientID   string
	httpClient *http.Client
}

// New returns a Twitch backend
func New(clientID string, logger logging.Logger) Client {
	twitchClient := &client{
		logger:     logger,
		clientID:   clientID,
		httpClient: &http.Client{},
	}

	return twitchClient
}

func (t *client) GetClientID() string {
	return t.clientID
}

func (t *client) ValidateToken(token string) (ValidateTokenResponse, error) {
	req, err := http.NewRequest("GET", "https://id.twitch.tv/oauth2/validate", nil)
	if err != nil {
		t.logger.Warnf("Error generating HTTP request %v", err)
		return ValidateTokenResponse{}, err
	}

	req.Header.Set("Authorization", fmt.Sprintf("OAuth %s", token))

	resp, err := t.httpClient.Do(req)
	if err != nil {
		t.logger.Warnf("Error performing HTTP request %v", err)
		return ValidateTokenResponse{}, err
	}

	if resp.StatusCode != http.StatusOK {
		t.logger.Warnf("Got non 200 status code from token validator %v", resp.StatusCode)
		return ValidateTokenResponse{}, errors.New("Unauthorized")
	}

	var result ValidateTokenResponse
	err = json.NewDecoder(resp.Body).Decode(&result)
	if err != nil {
		t.logger.Warnf("Error decoding response %v", err)
		return ValidateTokenResponse{}, err
	}

	return result, nil
}

func (t *client) GetChannel(token string) (GetChannelResponse, error) {
	req, err := http.NewRequest("GET", "https://api.twitch.tv/kraken/channel", nil)
	if err != nil {
		t.logger.Warnf("Error generating HTTP request %v", err)
		return GetChannelResponse{}, err
	}

	req.Header.Set("Authorization", fmt.Sprintf("OAuth %s", token))
	req.Header.Set("Accept", "application/vnd.twitchtv.v5+json")

	resp, err := t.httpClient.Do(req)
	if err != nil {
		t.logger.Warnf("Error performing HTTP request %v", err)
		return GetChannelResponse{}, err
	}

	if resp.StatusCode != http.StatusOK {
		t.logger.Warnf("Got non 200 status code from get channel %v", resp.StatusCode)
		return GetChannelResponse{}, errors.New("Unauthorized")
	}

	var result GetChannelResponse
	err = json.NewDecoder(resp.Body).Decode(&result)
	if err != nil {
		t.logger.Warnf("Error decoding response %v", err)
		return GetChannelResponse{}, err
	}

	return result, nil
}
