import acm = require('@aws-cdk/aws-certificatemanager');
import cf = require('@aws-cdk/aws-cloudfront');
import iam = require('@aws-cdk/aws-iam');
import lambda = require('@aws-cdk/aws-lambda');
import route53 = require('@aws-cdk/aws-route53');
import route53targets = require('@aws-cdk/aws-route53-targets');
import s3 = require('@aws-cdk/aws-s3');
import s3deploy = require('@aws-cdk/aws-s3-deployment');
import cdk = require('@aws-cdk/core');
import { LambdaEdgeEventType } from "@aws-cdk/aws-cloudfront";

export interface MoonlightAppProps {
  hostedZoneName: string
  appDomainName: string
  cspFunctionArn: string
}

export class MoonlightApp extends cdk.Construct {
  CloudFrontWebDistribution: cf.CloudFrontWebDistribution

  constructor(scope: cdk.Construct, id: string, props: MoonlightAppProps) {
    super(scope, id);

    const hostedZone = route53.HostedZone.fromLookup(this, 'MoonlightHostedZoneRef', {
      domainName: props.hostedZoneName,
    })

    const websiteBucket = new s3.Bucket(this, 'MoonlightAppBucket', {
      websiteIndexDocument: 'index.html',
      publicReadAccess: false,
    });

    const originId = new cf.OriginAccessIdentity(this, 'MoonlightAppOriginAccessIdentity', {
      comment: 'Origin access identity for the Moonlight App',
    });

    // Allow the CF Origin Access Identity access to the items in the bucket
    const originAccessPolicy = new iam.PolicyStatement({
      effect: iam.Effect.ALLOW,
      actions: ['s3:GetObject'],
      principals: [originId.grantPrincipal],
      resources: [websiteBucket.arnForObjects("*")]
    });

    websiteBucket.addToResourcePolicy(originAccessPolicy)

    const cert = new acm.DnsValidatedCertificate(this, 'MoonlightAppCertificate', {
      region: 'us-east-1', // Certificates for Cloudfront need to be in us-east-1
      domainName: props.appDomainName,
      hostedZone: hostedZone,
    });

    let lambdaAssociations = []

    if (props.cspFunctionArn != '') {
      lambdaAssociations.push({
        eventType: LambdaEdgeEventType.VIEWER_RESPONSE,
        lambdaFunction: lambda.Version.fromVersionArn(this, 'CSPFunctionArn', props.cspFunctionArn)
      })
    }

    this.CloudFrontWebDistribution = new cf.CloudFrontWebDistribution(this, 'MoonlightAppDistribution', {
      aliasConfiguration: {
        names: [props.appDomainName],
        acmCertRef: cert.certificateArn,
      },
      originConfigs: [{
        behaviors: [{
          isDefaultBehavior: true,
          lambdaFunctionAssociations: lambdaAssociations,
        }],
        s3OriginSource: {
          originAccessIdentity: originId,
          s3BucketSource: websiteBucket,
        },
      }],
      errorConfigurations: [{
        errorCode: 403,
        responseCode: 200,
        responsePagePath: '/index.html',
      },
      {
        errorCode: 404,
        responseCode: 200,
        responsePagePath: '/index.html',
      }],
    })

    new route53.ARecord(this, 'MoonlightAppARecord', {
      zone: hostedZone,
      recordName: '',
      target: route53.RecordTarget.fromAlias(new route53targets.CloudFrontTarget(this.CloudFrontWebDistribution))
    })

    new route53.AaaaRecord(this, 'MoonlightAppAaaaRecord', {
      zone: hostedZone,
      recordName: '',
      target: route53.RecordTarget.fromAlias(new route53targets.CloudFrontTarget(this.CloudFrontWebDistribution))
    })

    new s3deploy.BucketDeployment(this, 'DeployMoonlightApp', {
      sources: [s3deploy.Source.asset('../build')],
      destinationBucket: websiteBucket,
      distribution: this.CloudFrontWebDistribution,
    });
  }
}
