/**
 * This file is used to collect anonymous usage data of this project!
 * 
 * You can opt-out of data collection by setting the 
 * Node ENV variable "TRACKING_PROFILE_ID" to an empty string
 * and that will disable all tracking events! 
 * 
 * 
 * GA Event tracking params:
 * 
 * {
 *   ec: Event Category
 *   ea: Event Action
 *   el: Event Label
 *   ev: Event Value
 * }
 * 
 */

// Ensure environment variables are read.
process.env.NODE_ENV = 'development';
require('../../config/env');
const ua = require('universal-analytics');
const machineUuid = require("machine-uuid")
const chalk = require('chalk');
const googleTrackingID = process.env.TRACKING_PROFILE_ID;
const shouldTrack = !!(process.env.TRACKING_PROFILE_ID && process.env.TRACKING_PROFILE_ID.length > 0);

module.exports.printStatusMessage = function() {
  if (shouldTrack) {
    console.log(chalk.cyan("Tracking with Google Analytics is: Enabled \n"));
  } else {
    console.log(chalk.cyan("Tracking with Google Analytics is: Disabled \n"));
  }
}

module.exports.setupWebpackTracking = async function(webpackCompiler) {

  if (!shouldTrack) { return; }

  const analytics = await setupAnalytics();

  let firstCompileDone = false;

  webpackCompiler.plugin('done', (stats) => {
    analytics.event({
      ec: 'Server',
      ea: 'Compiled',
      el: firstCompileDone ? 'First Compile' : 'Re-Compile',
      ev: stats.endTime - stats.startTime, // compile time in ms
    }).send();
    firstCompileDone = true;
  });
}

async function track(action) {
  if (!shouldTrack) {
    return;
  }

  const analytics = await setupAnalytics();

  switch(action) {
    case 'start':
      analytics.event({
        ec: 'Server',
        ea: 'Started',
      }).send();
      break;
    case 'deploy': 
      analytics.event({
        ec: 'Deploy',
        ea: 'yarn deploy',
      }).send();
      break;
  }
}
module.exports.track = track;

async function setupAnalytics() {
  const uuid = await machineUuid();
  return ua(googleTrackingID, uuid, {strictCidFormat: false});
}

// If this was called directly with node, take some actions
if (process.argv && process.argv[2]) {
  track(process.argv[2]);
}
