import { ClickOutDetector } from "moonlight/components/click-out-detector";
import { userManager } from 'moonlight/components/oidc/userManager';
import { InstanceInfo } from 'moonlight/components/stream-control/reducers';
import { User } from 'oidc-client';
import * as React from "react";
import {
  AlignItems,
  Background,
  ChannelStatusIndicator,
  ChannelStatusIndicatorStatus,
  Color,
  CoreText,
  Display,
  Interactable,
  JustifyContent,
  Layout,
  Placeholder,
  Position,
  StyledLayout,
  SVG,
  SVGAsset
} from "twitch-core-ui";
import "./nav-bar.scss";
import { UserMenuDropdown } from "./user-menu-dropdown";

export interface PublicProps { }

export interface ReduxStateProps {
  user: User;
  instanceInfo: InstanceInfo | null;
}

export interface ReduxDispatchProps {
}

export type Props = PublicProps & ReduxStateProps & ReduxDispatchProps;

interface State {
  dropdownOpen: boolean;
}

export class NavBar extends React.Component<Props, State> {
  constructor(props: Props, state: State) {
    super(props, state);

    this.state = {
      dropdownOpen: false
    };
  }

  public render() {
    let placeholder = (
      <Layout
        flexGrow={0}
        flexShrink={0}
        display={Display.Flex}
        alignItems={AlignItems.Start}
      >
        <Layout padding={{ left: 0.5 }}>
          <Placeholder overlay width={200} height={14} />
          <Placeholder overlay lineCount={1} />
        </Layout>
      </Layout>
    );

    let userDropDown = placeholder;

    if (this.props && this.props.user) {
      let isStreaming = this.props.instanceInfo && this.props.instanceInfo.heartbeat && this.props.instanceInfo.heartbeat.streaming;

      let userName = (<CoreText><ChannelStatusIndicator status={isStreaming ? ChannelStatusIndicatorStatus.Live : ChannelStatusIndicatorStatus.Offline} pulse={isStreaming} /> {this.props.user.profile.preferred_username}</CoreText>);

      let instanceStatus = (<CoreText>Awaiting Status</CoreText>);

      if (this.props.instanceInfo) {
        instanceStatus = (<CoreText>{this.props.instanceInfo.status}</CoreText>);
      }

      userDropDown = (
        <Layout
          flexGrow={0}
          flexShrink={0}
          display={Display.Flex}
          alignItems={AlignItems.Start}
        >
          <Interactable onClick={this.onUserMenuClick} blurAfterClick >
            <Layout
              display={Display.Flex}
              flexGrow={1}
              alignItems={AlignItems.Stretch}
              fullHeight
              padding={1}
            >
              <Layout padding={{ left: 0.5 }} className="top-nav_user-info">
                <StyledLayout>
                  <CoreText>{userName}{instanceStatus}</CoreText>
                </StyledLayout>
              </Layout>
              <StyledLayout
                display={Display.Flex}
                flexShrink={0}
                color={Color.Base}
              >
                <SVG asset={SVGAsset.GlyphArrDown} />
              </StyledLayout>
            </Layout>
          </Interactable>
        </Layout>
      );
    }

    return (
      <StyledLayout
        display={Display.Flex}
        alignItems={AlignItems.Center}
        justifyContent={JustifyContent.Between}
        background={Background.Accent}
        color={Color.Overlay}
      >
        <Layout flexGrow={0} flexShrink={0} padding={1} >
          <SVG asset={SVGAsset.LogoGlitch} width={26} height={26} />
        </Layout>
        <Layout
          position={Position.Relative}
          display={Display.Flex}
          flexGrow={0}
          alignItems={AlignItems.Stretch}
          fullHeight
          className="tw-root--theme-dark"
        >
          <ClickOutDetector onClickOut={this.onUserMenuClickOut}>
            {userDropDown}
            <UserMenuDropdown
              dropdownOpen={this.state.dropdownOpen}
              closeDropdown={this.onUserMenuClick}
              logout={this.logout}
            />
          </ClickOutDetector>
        </Layout>
      </StyledLayout>
    );
  }

  private onUserMenuClickOut = () => {
    this.setState({ dropdownOpen: false });
  }

  private onUserMenuClick = () => {
    if (!this.props.user) {
      return;
    }
    this.setState({ dropdownOpen: !this.state.dropdownOpen });
  }

  private logout = () => {
    this.setState({
      dropdownOpen: false
    });
    userManager.removeUser();
  }
}
