import { API } from 'moonlight/api/api';
import { APIResponse } from 'moonlight/api/apiresponse';
import { GlobalState } from 'moonlight/utils/reduxStore';
import { Dispatch } from 'redux';
import { InitResponse, InstanceInfo, StreamConfig } from './reducers';

export enum StreamControlActions {
  FetchStreamConfig = 'STREAMCONTROL_FETCH_STREAM_CONFIG',
  SetStreamConfig = 'STREAMCONTROL_SET_STREAM_CONFIG',
  SetCurrentScene = 'STREAMCONTROL_SET_CURRENT_SCENE',
  SetLiveness = 'STREAMCONTROL_SET_LIVENESS',
  SetLivenessState = 'STREAMCONTROL_SET_LIVENESS_STATE',
  SetMuteState = 'STREAMCONTROL_SET_MUTE_STATE',
  SetInstanceInfo = 'STREAMCONTROL_SET_INSTANCE_INFO'
}

export const setStreamConfig = (item: StreamConfig) => {
  return  {
    type: StreamControlActions.SetStreamConfig,
    data: {
      ...item,
    },
  };
};

export const setInstanceInfo = (item: InstanceInfo) => {
  return  {
    type: StreamControlActions.SetInstanceInfo,
    data: {
      ...item,
    },
  };
};

export const setCurrentScene = (item: string) => {
  return  {
    type: StreamControlActions.SetCurrentScene,
    data: item,
  };
};

export const setLivenessState = (item: boolean) => {
  return  {
    type: StreamControlActions.SetLivenessState,
    data: item,
  };
};

export const setMuteState = (item: string) => {
  return  {
    type: StreamControlActions.SetMuteState,
    data: item,
  };
};

export const fetchStreamConfig = (): any => {
  return async (dispatch: Dispatch, getState: () => GlobalState) => {
    const state = getState();
    let promise = API.get<InitResponse>("/init", state.oidc.user.access_token, {});

    // Get new value
    return promise.then((response: APIResponse<InitResponse>) => {
      if (response.status != 200) {
        dispatch(setStreamConfig({
          config_found: false,
          scenes: [],
        }));
      } else {
        let resp = response.body as InitResponse;

        dispatch(setStreamConfig({
          config_found: true,
          scenes: resp.config.scenes,
        }));

        dispatch(setInstanceInfo(resp.instance_info));
      }
    });
  };
};

interface RawInstanceInfo {
  status: string;
  rtmp_muted: boolean;
  heartbeat_json: string;
}

export const fetchInstanceStatus = (): any => {
  return async (dispatch: Dispatch, getState: () => GlobalState) => {
    const state = getState();
    let promise = API.get<RawInstanceInfo>("/status", state.oidc.user.access_token, {});

    // Get new value
    return promise.then((response: APIResponse<RawInstanceInfo>) => {
      if (response.status != 200) {
        dispatch(setInstanceInfo({
          status: response.status == 404 ? "NOT FOUND" : "ERRORED",
          rtmp_muted: false,
          heartbeat: undefined,
        }));
      } else {
        let rawInstanceInfo = response.body as RawInstanceInfo;

        let instanceInfo = {
          status: rawInstanceInfo.status,
          rtmp_muted: rawInstanceInfo.rtmp_muted,
          heartbeat: rawInstanceInfo.heartbeat_json ? JSON.parse(rawInstanceInfo.heartbeat_json) : undefined,
        };

        dispatch(setInstanceInfo(instanceInfo));
      }
    });
  };
};

export const setLiveness = (live: boolean): any => {
  return async (dispatch: Dispatch, getState: () => GlobalState) => {
    const state = getState();
    let promise = API.put<boolean>("/live", state.oidc.user.access_token, { body: { live } });

    // Get new value
    return promise.then((response: any) => {
      dispatch(
        setLivenessState(response.body),
      );
      return true;
    }, () => {
      return false;
    });
  };
};

export const setScene = (sceneIdentifier: string): any => {
  return async (dispatch: Dispatch, getState: () => GlobalState) => {
    const state = getState();
    let promise = API.put<string>("/current-scene", state.oidc.user.access_token, { body: { scene_identifier: sceneIdentifier } });

    // Get new value
    return promise.then((response: any) => {
      dispatch(
        setCurrentScene(response.body),
      );
      return true;
    }, () => {
      return false;
    });
  };
};

export const setMute = (muted: boolean): any => {
  return async (dispatch: Dispatch, getState: () => GlobalState) => {
    const state = getState();
    let promise = API.put<boolean>("/mute", state.oidc.user.access_token, { body: { mute: muted } });

    // Get new value
    return promise.then((response: any) => {
      dispatch(
        setMuteState(response.body),
      );
      return true;
    }, () => {
      return false;
    });
  };
};
