import { MassiveButton } from 'moonlight/components/massive-button';
import * as React from "react";
import {
  CoreText,
  Display,
  FlexDirection,
  Layout,
  LoadingSpinner,
  TextAlign,
  TextType,
  Typeset
} from "twitch-core-ui";
import { InstanceInfo, InstanceStatus, StreamConfig } from './reducers';
import "./stream-control.scss";

export interface PublicProps { }

export interface ReduxStateProps {
  config: StreamConfig;
  instanceInfo: InstanceInfo;
}

export interface ReduxDispatchProps {
  setScene: (sceneIdentifier: string) => Promise<boolean>;
  setLiveness: (live: boolean) => Promise<boolean>;
  setMute: (muted: boolean) => Promise<boolean>;
  fetchStreamConfig: () => void;
  fetchInstanceStatus: () => void;
}

export type Props = PublicProps & ReduxStateProps & ReduxDispatchProps;

interface State {
  currentPanel: string;
}

export enum CurrentPanel {
  Scenes = "SCENES",
  Audio = "AUDIO",
  Stream = "STREAM",
}

export class StreamControl extends React.Component<Props, State> {
  private intervalToken: number;

  constructor(props: Props, state: State) {
    super(props, state);

    this.state = {
      currentPanel: CurrentPanel.Scenes
    };
  }

  public componentWillMount() {
    this.props.fetchStreamConfig();

    this.intervalToken = window.setInterval(() => this.getStatus(), 1000);
  }

  public componentWillUnmount() {
    window.clearInterval(this.intervalToken);
  }

  public render() {
    let content = <Layout display={Display.InlineBlock}><LoadingSpinner /></Layout>;

    let notFound = <CoreText type={TextType.H3}>Your account is not set up, if you believe this is in error please talk to your Twitch representative.</CoreText>;

    if (this.props.config) {
      if (this.props.config.config_found) {
        if (this.props.instanceInfo != null) {
          if (this.props.instanceInfo.status == InstanceStatus.Healthy.toString()) {
            return this.layout();
          }

          if (this.props.instanceInfo.status == InstanceStatus.Running.toString()) {
            return this.basicLayout('Instance has started, awaiting configuration to be complete');
          }

          if (this.props.instanceInfo.status == InstanceStatus.Unhealthy.toString()) {
            return this.basicLayout('Instance unhealthy, this may be because it has just started in which case, just wait for a few moments, but if this problem persists you may need to restart the instance');
          }

          if (this.props.instanceInfo.status == InstanceStatus.NotFound.toString()) {
            return this.basicLayout('Instance has been stopped, please reload to generate a new instance');
          }
        }

        return this.basicLayout('Waiting for instance to load');
      }

      content = notFound;
    }

    return (
      <Layout padding={3} textAlign={TextAlign.Center}>
        <Typeset>
          {content}
        </Typeset>
      </Layout>
    );
  }

  private getStatus() {
    if (!this.props.config || !this.props.config.config_found) {
      return;
    }

    // Not found requires a reload back to init
    if (this.props.instanceInfo.status == InstanceStatus.NotFound.toString()) {
      clearInterval(this.intervalToken);
      this.intervalToken = undefined;
      return;
    }

    this.props.fetchInstanceStatus();
  }

  private scenes() {
    return this.props.config.scenes.map((scene) => (
      <Layout key={scene} display={Display.Flex} flexGrow={1}>
        <MassiveButton onClick={() => this.props.setScene(scene)} title={scene} selected={this.props.instanceInfo.heartbeat && this.props.instanceInfo.heartbeat["current-scene"] == scene} />
      </Layout>
    ));
  }

  private audio() {
    return (
      <>
        <Layout display={Display.Flex} flexGrow={1}>
          <MassiveButton onClick={() => this.props.setMute(true)} title={"Mute"} selected={this.props.instanceInfo.rtmp_muted} />
        </Layout>
        <Layout display={Display.Flex} flexGrow={1}>
          <MassiveButton onClick={() => this.props.setMute(false)} title={"UnMute"} selected={!this.props.instanceInfo.rtmp_muted} />
        </Layout>
      </>
    );
  }

  private stream() {
    return (
      <>
        <Layout display={Display.Flex} flexGrow={1}>
          <MassiveButton onClick={() => this.props.setLiveness(true)} title={"Start Stream"} selected={this.props.instanceInfo.heartbeat && this.props.instanceInfo.heartbeat.streaming} />
        </Layout>
        <Layout display={Display.Flex} flexGrow={1}>
          <MassiveButton onClick={() => this.props.setLiveness(false)} title={"Stop Stream"} selected={this.props.instanceInfo.heartbeat && !this.props.instanceInfo.heartbeat.streaming} />
        </Layout>
      </>
    );
  }

  private async setPanel(panel: CurrentPanel): Promise<boolean> {
    this.setState({
      currentPanel: panel,
    });

    return await true;
  }

  private layout() {
    let currentPanel = this.state ? this.state.currentPanel : null;

    return (
      <Layout fullHeight display={Display.Flex} flexDirection={FlexDirection.Column}>
        <Layout flexGrow={1} fullWidth display={Display.Flex} flexDirection={FlexDirection.Row}>
          <Layout display={Display.Flex} flexGrow={1}>
            <MassiveButton onClick={() => this.setPanel(CurrentPanel.Scenes)} title={"Scenes"} selected={currentPanel == CurrentPanel.Scenes} />
          </Layout>
          <Layout display={Display.Flex} flexGrow={1}>
            <MassiveButton onClick={() => this.setPanel(CurrentPanel.Audio)} title={"Audio"} selected={currentPanel == CurrentPanel.Audio} />
          </Layout>
          <Layout display={Display.Flex} flexGrow={1}>
            <MassiveButton onClick={() => this.setPanel(CurrentPanel.Stream)} title={"Stream"} selected={currentPanel == CurrentPanel.Stream} />
          </Layout>
        </Layout>
        <Layout flexGrow={4} fullWidth display={Display.Flex} flexDirection={FlexDirection.Column}>
          {(() => {
            switch (currentPanel) {
              case CurrentPanel.Scenes:
                return this.scenes();
              case CurrentPanel.Audio:
                return this.audio();
              case CurrentPanel.Stream:
                return this.stream();
              default:
                return null;
            }
          })()}
        </Layout>
      </Layout>
    );
  }

  private basicLayout(message: string) {
    return (
      <Layout textAlign={TextAlign.Center} padding={5}>
        <Typeset>
          <CoreText type={TextType.H3}>{message}</CoreText>
        </Typeset>
      </Layout>
    );
  }
}
