package main

import (
	"code.justin.tv/event-engineering/acm-ca-go/pkg/certgen"
	awsBackend "code.justin.tv/event-engineering/moonlight-daemon/pkg/aws/backend"
	"code.justin.tv/event-engineering/moonlight-daemon/pkg/moonlight"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/session"
	"github.com/aws/aws-sdk-go/service/ssm"
	"github.com/sirupsen/logrus"
	"os"
	"os/signal"
	//"strings"
	"flag"
	"path"
	"syscall"
)

var logRoot = flag.String("logdir", "/var/opt/moonlight/log", "Log directory location")
var tmpDir = flag.String("tmpdir", "/var/opt/moonlight/tmp", "Temp directory location")
var awsRegion = flag.String("region", "us-west-2", "AWS region")

func main() {
	logger := logrus.New()
	logger.SetLevel(logrus.DebugLevel)
	logger.SetNoLock()
	logger.SetFormatter(&logrus.JSONFormatter{})

	err := os.MkdirAll(*logRoot, os.ModeDir)
	if err != nil {
		panic(err)
	}

	log, err := os.OpenFile(path.Join(*logRoot, "daemon.log"), os.O_APPEND|os.O_WRONLY|os.O_CREATE, 0766)
	if err != nil {
		panic(err)
	}
	defer log.Close()

	logger.SetOutput(log)

	sess, err := session.NewSession(&aws.Config{
		Region: aws.String(*awsRegion),
	})

	ab, err := awsBackend.New(sess)
	if err != nil {
		panic(err)
	}

	var internalLambdaARN string
	internalLambdaARNParamName := "/moonlight/internalLambdaARN"

	var moonlightRootCAARN string
	moonlightRootCAARNParamName := "/moonlight/moonlightRootCAARN"

	params, err := ab.SSMGetParameters(&ssm.GetParametersInput{
		Names: []*string{
			&internalLambdaARNParamName,
			&moonlightRootCAARNParamName,
		},
	})

	if err != nil {
		panic(err)
	}

	for _, param := range params.Parameters {
		switch *param.Name {
		case internalLambdaARNParamName:
			internalLambdaARN = *param.Value
			break
		case moonlightRootCAARNParamName:
			moonlightRootCAARN = *param.Value
			break
		}
	}

	certGen := certgen.New(sess, moonlightRootCAARN)

	server, err := moonlight.New(internalLambdaARN, *logRoot, *tmpDir, ab, certGen, logger)

	if err != nil {
		panic(err)
	}

	server.Start()

	defer func() {
		if r := recover(); r != nil {
			logger.Errorf("Caught panic %v", r)
		}
		logger.Info("Stopping server")
		server.Stop(false)
		logger.Info("Server stopped, exiting")
	}()

	osSig := make(chan os.Signal, 1)
	signal.Notify(osSig, os.Interrupt, syscall.SIGTERM, syscall.SIGQUIT)

	sig := <-osSig

	logger.Infof("Received signal %v", sig)
}
