package moonlight

import (
	"code.justin.tv/event-engineering/moonlight-daemon/pkg/rpc"
	"context"
	"crypto/tls"
	"errors"
	"fmt"
	"net"
	"net/http"
)

var listenPort = 8396
var mutableSource = "RTMP"

func (s *server) configureListener(listenIP, tlsHost string) {
	s.wg.Add(1)

	twirpHandler := rpc.NewDaemonServer(s, nil)

	tlsCert, err := s.certGen.CanHazCert(tlsHost, nil)
	if err != nil {
		s.logger.Errorf("Could not create TLS certificate: %v\n", err)
		s.wg.Done()
	}

	tlsConfig := &tls.Config{
		Certificates: []tls.Certificate{tlsCert},
	}

	server := &http.Server{
		Addr:      fmt.Sprintf("%s:%v", tlsHost, listenPort),
		TLSConfig: tlsConfig,
		Handler:   twirpHandler,
	}

	listener, err := net.Listen("tcp4", fmt.Sprintf("%s:%v", listenIP, listenPort))
	if err != nil {
		s.logger.Errorf("Could not create daemon listener %v", err)
		s.wg.Done()
	}

	s.listener = listener

	go func() {
		err = server.ServeTLS(s.listener, "", "")

		if err != nil {
			if !s.stopped {
				s.logger.Errorf("Daemon listener error %v", err)
			} else {
				s.logger.Debug("Daemon listener closed")
			}

			s.wg.Done()
		}
	}()
}

func (s *server) stopListener() {
	if s.listener != nil {
		s.listener.Close()
	}
}

func (s *server) AllocateInstance(ctx context.Context, req *rpc.AllocateInstanceReq) (*rpc.AllocateInstanceResp, error) {
	if s.stopped {
		return nil, errors.New("Server is stopping")
	}

	slot, err := s.allocateSlot(req.GetInstanceId())
	if err != nil {
		s.logger.Warnf("Failed to allocate slot, %v", err)
		return nil, err
	}

	s.logger.Debugf("Allocated slot for instance id %v - debug port %v", req.InstanceId, slot.VNCDebugPort)

	return &rpc.AllocateInstanceResp{}, nil
}

func (s *server) GetStatus(ctx context.Context, req *rpc.GetStatusReq) (*rpc.GetStatusResp, error) {
	if s.stopped {
		return nil, errors.New("Server is stopping")
	}

	if instance, ok := s.instances[req.InstanceId]; ok {
		status, err := instance.GetOBSServer().GetStatus()
		if err != nil {
			return &rpc.GetStatusResp{
				Success:            true,
				CurrentScene:       status.CurrentScene,
				CurrentBitrateKbps: int32(status.CurrentBitrateKbps),
				IsStreaming:        status.IsStreaming,
			}, nil
		}

		return &rpc.GetStatusResp{
			Success: false,
			Message: "Unable to retrieve status",
		}, nil
	}

	return nil, errors.New("Invalid instance ID")
}

func (s *server) StartStream(ctx context.Context, req *rpc.StartStreamReq) (*rpc.StartStreamResp, error) {
	if s.stopped {
		return nil, errors.New("Server is stopping")
	}

	if instance, ok := s.instances[req.InstanceId]; ok {
		err := instance.GetOBSServer().StartStream(req.GetStreamKey())
		if err != nil {
			return &rpc.StartStreamResp{
				Success: false,
				Message: "Error starting stream",
			}, nil
		}

		return &rpc.StartStreamResp{
			Success: true,
		}, nil
	}
	return nil, errors.New("Invalid instance ID")
}

func (s *server) StopStream(ctx context.Context, req *rpc.StopStreamReq) (*rpc.StopStreamResp, error) {
	if s.stopped {
		return nil, errors.New("Server is stopping")
	}

	if instance, ok := s.instances[req.InstanceId]; ok {
		err := instance.GetOBSServer().StopStream()
		if err != nil {
			return &rpc.StopStreamResp{
				Success: false,
				Message: "Error stopping stream",
			}, nil
		}

		return &rpc.StopStreamResp{
			Success: true,
		}, nil
	}
	return nil, errors.New("Invalid instance ID")
}

// I don't think we need this any more
func (s *server) GetSceneList(ctx context.Context, req *rpc.GetSceneListReq) (*rpc.GetSceneListResp, error) {
	if s.stopped {
		return nil, errors.New("Server is stopping")
	}

	if instance, ok := s.instances[req.InstanceId]; ok {
		scenes, err := instance.GetOBSServer().GetSceneList()
		if err != nil {
			return &rpc.GetSceneListResp{
				Success: false,
				Message: "Error retrieving scene list",
			}, nil
		}

		return &rpc.GetSceneListResp{
			Success:      true,
			CurrentScene: scenes.CurrentScene,
			Scenes:       scenes.Scenes,
		}, nil
	}
	return nil, errors.New("Invalid instance ID")
}

func (s *server) SwitchScene(ctx context.Context, req *rpc.SwitchSceneReq) (*rpc.SwitchSceneResp, error) {
	if s.stopped {
		return nil, errors.New("Server is stopping")
	}

	if instance, ok := s.instances[req.InstanceId]; ok {
		err := instance.GetOBSServer().SwitchScene(req.SceneIdentifier)
		if err != nil {
			return &rpc.SwitchSceneResp{
				Success: false,
				Message: "Error switching scene",
			}, nil
		}

		return &rpc.SwitchSceneResp{
			Success: true,
		}, nil
	}
	return nil, errors.New("Invalid instance ID")
}

func (s *server) GetVNCKey(ctx context.Context, req *rpc.GetVNCKeyReq) (*rpc.GetVNCKeyResp, error) {
	if s.stopped {
		return nil, errors.New("Server is stopping")
	}

	if instance, ok := s.instances[req.InstanceId]; ok {
		vncKey, vncPort, err := instance.GetVNCKey()
		if err != nil {
			s.logger.Errorf("Error retrieving VNC key %v", err)
			return &rpc.GetVNCKeyResp{
				Success: false,
				Message: "Error getting VNC key",
			}, nil
		}

		return &rpc.GetVNCKeyResp{
			Success: true,
			VncKey:  vncKey,
			VncPort: vncPort,
		}, nil
	}

	return nil, errors.New("Invalid instance ID")
}

func (s *server) StopInstance(ctx context.Context, req *rpc.StopInstanceReq) (*rpc.StopInstanceResp, error) {
	if s.stopped {
		return nil, errors.New("Server is stopping")
	}

	s.logger.Infof("Got request to stop instance with id %v", req.InstanceId)

	s.removeInstance(req.InstanceId, true)

	return &rpc.StopInstanceResp{
		Success: true,
	}, nil
}

func (s *server) MuteInput(ctx context.Context, req *rpc.MuteInputReq) (*rpc.MuteInputResp, error) {
	if s.stopped {
		return nil, errors.New("Server is stopping")
	}

	if instance, ok := s.instances[req.InstanceId]; ok {
		err := instance.GetOBSServer().MuteInput(mutableSource, req.GetMute())
		if err != nil {
			return &rpc.MuteInputResp{
				Success: false,
				Message: "Error setting mute state on source",
			}, nil
		}

		return &rpc.MuteInputResp{
			Success: true,
		}, nil
	}
	return nil, errors.New("Invalid instance ID")
}

func (s *server) UpdateOBSConfig(ctx context.Context, req *rpc.UpdateOBSConfigReq) (*rpc.UpdateOBSConfigResp, error) {
	if s.stopped {
		return nil, errors.New("Server is stopping")
	}

	if instance, ok := s.instances[req.InstanceId]; ok {
		err := instance.GetOBSServer().UpdateOBSConfig(req.MainOverlayUrl, int(req.MainOverlayWidth), int(req.MainOverlayHeight), int(req.MainOverlayX), int(req.MainOverlayY))
		if err != nil {
			s.logger.Warnf("Error updating OBS config %v", err)
			return &rpc.UpdateOBSConfigResp{
				Success: false,
				Message: "Error updating OBS config",
			}, nil
		}

		return &rpc.UpdateOBSConfigResp{
			Success: true,
		}, nil
	}
	return nil, errors.New("Invalid instance ID")
}

func (s *server) Terminate(ctx context.Context, req *rpc.TerminateDaemonReq) (*rpc.TerminateDaemonResp, error) {
	if len(s.instances) > 0 {
		return &rpc.TerminateDaemonResp{
			Success: false,
			Message: "Daemon has active instances",
		}, nil
	}

	s.Stop(true)

	return &rpc.TerminateDaemonResp{
		Success: true,
	}, nil
}
