package moonlight

import (
	"code.justin.tv/event-engineering/golibs/pkg/logging"
	rpc "code.justin.tv/event-engineering/moonlight-api/pkg/rpc"
	"code.justin.tv/event-engineering/moonlight-daemon/pkg/aws"
	"code.justin.tv/event-engineering/moonlight-daemon/pkg/obs"
	"code.justin.tv/event-engineering/moonlight-daemon/pkg/xorg"
	obsws "code.justin.tv/event-engineering/obs-websocket-go/pkg/obs"
	"encoding/base64"
	"errors"
	"fmt"
	"github.com/aws/aws-sdk-go/service/ecr"
	dockerClient "github.com/fsouza/go-dockerclient"
	"os"
	"os/user"
	"path"
	"regexp"
	"strconv"
	"strings"
	"syscall"
	"time"
)

var (
	vncKeyRegex             = regexp.MustCompile(`^\s*Full control one-time password: ([0-9]+)\s*$`)
	heartbeatUpdateInterval = 20 * time.Second
)

type Instance interface {
	Start(config *rpc.OBSCloudInstance, pciBusID string) error
	Stop(stopInstance bool) error
	GetVNCKey() (string, int32, error)
	GetOBSServer() obs.Server
	IsStarted() bool
	GetLastHeartbeat() InstanceStatus
}

type InstanceStatus struct {
	LastUpdated  time.Time
	LastStreamed time.Time
	RTMPMuted    bool
	Heartbeat    obsws.EventHeartbeat
}

type instance struct {
	logger                     logging.Logger
	obs                        obs.Server
	containerID                string
	docker                     *dockerClient.Client
	aws                        *aws.Client
	slot                       Slot
	stopped                    bool
	logDir                     string
	tmpDir                     string
	dockerRegistryURL          string
	lastHeartbeat              InstanceStatus
	isStarted                  bool
	moonlightBridgeNetworkName string
}

func newInstance(slot Slot, docker *dockerClient.Client, aws *aws.Client, logDir, tmpDir, moonlightBridgeNetworkName string, logger logging.Logger) Instance {
	return &instance{
		logger: logger,
		docker: docker,
		aws:    aws,
		slot:   slot,
		logDir: logDir,
		tmpDir: tmpDir,
		lastHeartbeat: InstanceStatus{
			LastUpdated:  time.Now(),
			LastStreamed: time.Now(),
		},
		moonlightBridgeNetworkName: moonlightBridgeNetworkName,
	}
}

func (i *instance) IsStarted() bool {
	return i.isStarted
}

func (i *instance) GetLastHeartbeat() InstanceStatus {
	return i.lastHeartbeat
}

func generateNetworkName(networkID int32) string {
	return fmt.Sprintf("moonlight-bridge-%v", networkID)
}

func (i *instance) Start(config *rpc.OBSCloudInstance, pciBusID string) error {
	// Generate config
	configDir := path.Join(i.tmpDir, config.GetId())
	err := os.MkdirAll(configDir, 0766)
	if err != nil {
		i.logger.Warnf("Error creating config directory %v", err)
		return err
	}

	logDir := fmt.Sprintf(path.Join(i.logDir, "instances", config.GetId()))

	// Create log directory
	err = os.MkdirAll(logDir, 0766)
	if err != nil {
		i.logger.Warnf("Error generating log directory %v", err)
		return err
	}

	group, err := user.LookupGroup("adm")
	if err != nil {
		i.logger.Warnf("Error looking up adm user info %v", err)
	} else {
		gid, _ := strconv.Atoi(group.Gid)

		err = syscall.Chown(logDir, -1, gid)
		if err != nil {
			i.logger.Warnf("Error setting permissions on log directory %v", err)
		}

		err = syscall.Chown(configDir, -1, gid)
		if err != nil {
			i.logger.Warnf("Error setting permissions on config directory %v", err)
		}
	}

	obsConfigDir := path.Join(configDir, "obs")
	err = os.MkdirAll(obsConfigDir, 0766)
	if err != nil {
		i.logger.Warnf("Error creating config directory %v", err)
		return err
	}

	err = obs.GenerateConfig(obsConfigDir, config)
	if err != nil {
		i.logger.Warnf("Error generating OBS config %v", err)
		return err
	}

	xorgConfigDir := path.Join(configDir, "xorg")
	err = os.MkdirAll(xorgConfigDir, 0766)
	if err != nil {
		i.logger.Warnf("Error creating config directory %v", err)
		return err
	}

	err = xorg.GenerateXOrgConfig(xorgConfigDir, pciBusID)
	if err != nil {
		i.logger.Warnf("Error generating XOrg config %v", err)
		return err
	}

	// Get ECR registry creds
	creds, err := i.aws.ECRGetAuthorizationToken(&ecr.GetAuthorizationTokenInput{})
	if err != nil {
		i.logger.Warnf("Could not retrieve ECR credentials %v", err)
		return err
	}

	if len(creds.AuthorizationData) < 1 {
		i.logger.Warn("Could not retrieve ECR credentials, no results returned")
		return errors.New("No ECR credentials found")
	}

	token := *creds.AuthorizationData[0].AuthorizationToken
	bytes, err := base64.StdEncoding.DecodeString(token)
	if err != nil {
		i.logger.Warnf("Could not decode token %v", err)
		return err
	}

	tokenParts := strings.Split(string(bytes), ":")
	user := tokenParts[0]
	password := tokenParts[1]

	// Run docker container
	err = i.docker.PullImage(dockerClient.PullImageOptions{
		Repository: config.GetDockerImageName(),
		Tag:        "latest",
	}, dockerClient.AuthConfiguration{
		Username: user,
		Password: password,
	})

	if err != nil {
		i.logger.Warnf("Error pulling docker image %v | %v", config.GetDockerImageName(), err)
		return err
	}

	listResponse, err := i.docker.ListContainers(dockerClient.ListContainersOptions{
		All: true,
		Filters: map[string][]string{
			"name": []string{config.GetId()},
		},
	})

	if err != nil {
		return err
	}

	i.logger.Infof("Found %v containers with name %v", len(listResponse), config.GetId())

	for _, container := range listResponse {
		contanerErr := i.docker.RemoveContainer(dockerClient.RemoveContainerOptions{
			ID: container.ID,
		})

		if err != nil {
			return contanerErr
		}
	}

	i.logger.Infof("Creating new container with name %v", config.GetId())

	resp, err := i.docker.CreateContainer(dockerClient.CreateContainerOptions{
		Name: config.GetId(),
		Config: &dockerClient.Config{
			Labels: map[string]string{
				"MOONLIGHT":     "MOONLIGHT",
				"VNCDebugPort":  fmt.Sprint(i.slot.VNCDebugPort),
				"OBSRemotePort": fmt.Sprint(i.slot.OBSRemotePort),
				"DockerNetwork": fmt.Sprint(i.slot.DockerNetwork),
				"TTYNum":        fmt.Sprint(i.slot.TTYNum),
				"InstanceID":    config.GetId(),
			},
			Image: config.GetDockerImageName(),
			ExposedPorts: map[dockerClient.Port]struct{}{
				"5901": struct{}{},
				"4444": struct{}{},
			},
		},
		HostConfig: &dockerClient.HostConfig{
			Runtime:     "nvidia",
			NetworkMode: i.moonlightBridgeNetworkName,
			CapAdd: []string{
				"NET_ADMIN", // Allow the container to create iptables rules to prevent browser source from being able to access local ports for OBS remote and VNC
			},
			AutoRemove: true,
			PortBindings: map[dockerClient.Port][]dockerClient.PortBinding{
				dockerClient.Port("5901"): {{HostIP: "127.0.0.1", HostPort: fmt.Sprint(i.slot.VNCDebugPort)}},
				dockerClient.Port("4444"): {{HostIP: "127.0.0.1", HostPort: fmt.Sprint(i.slot.OBSRemotePort)}},
			},
			Devices: []dockerClient.Device{
				dockerClient.Device{
					PathOnHost:        fmt.Sprintf("/dev/tty%v", i.slot.TTYNum),
					PathInContainer:   "/dev/tty2",
					CgroupPermissions: "rwm",
				},
			},
			Memory:  2147483648, // 2GB
			ShmSize: 536870912,  // 512 MB
			Mounts: []dockerClient.HostMount{
				dockerClient.HostMount{
					Type:     "bind",
					Source:   obsConfigDir,
					Target:   "/home/obs/.config/obs-studio",
					ReadOnly: false,
				},
				dockerClient.HostMount{
					Type:     "bind",
					Source:   path.Join(xorgConfigDir, "xorg.conf"),
					Target:   "/etc/X11/xorg.conf",
					ReadOnly: false,
				},
				dockerClient.HostMount{
					Type:     "bind",
					Source:   logDir,
					Target:   "/var/log/moonlight",
					ReadOnly: false,
				},
			},
		},
	})

	if err != nil {
		i.logger.Warnf("Error creating docker container %v", err)
		return err
	}

	i.logger.Infof("Created new container with name %v id %v", config.GetId(), resp.ID)

	if err = i.docker.StartContainer(resp.ID, nil); err != nil {
		i.logger.Warnf("Error starting docker container %v", err)
		return err
	}

	i.containerID = resp.ID
	i.logger.Infof("Container %v started", resp.ID)

	i.startOBSConnection()

	return nil
}

func (i *instance) handleOBSHeartbeat(evt *obsws.EventHeartbeat) {
	muted, err := i.obs.GetMuteStatus(mutableSource)
	if err != nil {
		i.logger.Warnf("Error getting mute status %v", err)
	}

	lastStreamed := i.lastHeartbeat.LastStreamed
	if evt.Streaming {
		lastStreamed = time.Now()
	}

	i.lastHeartbeat = InstanceStatus{
		LastUpdated:  time.Now(),
		LastStreamed: lastStreamed,
		RTMPMuted:    muted,
		Heartbeat:    *evt,
	}
}

func (i *instance) handleEvent(evt obsws.Event) {
	switch evt.Type() {
	case "Heartbeat":
		i.handleOBSHeartbeat(evt.(*obsws.EventHeartbeat))
		break
	}
}

func (i *instance) startOBSConnection() {
	firstSuccess := true

	go func() {
		var attempt = 0

		for {
			if i.stopped {
				return
			}

			// Set started to true after 3 attempts so health checks will pick up on the unhealthy instance
			if attempt > 3 {
				i.isStarted = true
			}

			time.Sleep(5 * time.Second)
			attempt++

			i.logger.Debugf("Attempting to connect to OBS, attempt %v", attempt)

			port := i.slot.OBSRemotePort
			i.obs = obs.NewRemoteClient("127.0.0.1", int(port), i.logger)

			errCh, err := i.obs.Connect()
			if err != nil {
				i.logger.Errorf("OBS Remote client error, %v", err)
				i.obs.Close()
				continue
			}

			err = i.obs.SetHeartbeat(true)
			if err != nil {
				i.logger.Errorf("Failed to configure websocket heartbeat, %v", err)
				i.obs.Close()
				continue
			}

			eventCh := i.obs.Subscribe()

			go func() {
				for {
					evt := <-eventCh
					if evt == nil {
						return
					}
					go i.handleEvent(evt)
				}
			}()

			// We succeeded so set isStarted = true to start tracking health
			if firstSuccess {
				firstSuccess = false
				i.isStarted = true
			}

			err = <-errCh
			if err != nil {
				i.logger.Errorf("OBS Remote client error, %v", err)
			}
		}
	}()
}

func (i *instance) GetVNCKey() (string, int32, error) {
	if i.stopped {
		return "", 0, errors.New("Instance is stopped")
	}

	exec, err := i.docker.CreateExec(dockerClient.CreateExecOptions{
		Cmd:          []string{"vncpasswd", "-o", "-display", "unix:1"},
		AttachStdout: true,
		AttachStderr: true,
		Container:    i.containerID,
	})

	if err != nil {
		return "", 0, err
	}

	var writer strings.Builder

	err = i.docker.StartExec(exec.ID, dockerClient.StartExecOptions{
		OutputStream: &writer,
		ErrorStream:  &writer, // Because apparently this come back on stderr
	})

	if err != nil {
		return "", 0, err
	}

	output := writer.String()

	if !vncKeyRegex.MatchString(output) {
		return "", 0, fmt.Errorf("Unexpected command output %v", output)
	}

	vncKey := vncKeyRegex.ReplaceAllString(output, "$1")

	return vncKey, i.slot.VNCDebugPort, nil
}

func (i *instance) Stop(stopInstance bool) error {
	i.stopped = true

	// Validate stop, probably should return error if stream is running
	// unless some force close param is present perhaps?
	if i.obs != nil {
		i.obs.Close()
		i.obs = nil
	}

	if stopInstance {
		// Stop Docker container
		err := i.docker.StopContainer(i.containerID, 10)
		if err != nil {
			i.logger.Warnf("Failed to stop container %v", err)
		}
	}

	return nil
}

func (i *instance) GetOBSServer() obs.Server {
	return i.obs
}

func (s *server) populateRunningInstances() error {
	// Query docker to see if the container is running and grab it if so
	containers, err := s.docker.ListContainers(dockerClient.ListContainersOptions{})

	if err != nil {
		return err
	}

	s.instanceLock.Lock()
	defer s.instanceLock.Unlock()

	for _, container := range containers {
		if _, ok := container.Labels["MOONLIGHT"]; !ok {
			s.logger.Warnf("Found non-moonlight container with ID %v", container.ID)
			continue
		}
		instanceID := container.Labels["InstanceID"]

		ttyNum, err := strconv.ParseInt(container.Labels["TTYNum"], 10, 32)
		if err != nil {
			s.logger.Warnf("Failed to parse TTY Num from running container %v - %v", container.ID, container.Labels["TTYNum"])
			continue
		}
		obsPort, err := strconv.ParseInt(container.Labels["OBSRemotePort"], 10, 32)
		if err != nil {
			s.logger.Warnf("Failed to parse OBS port from running container %v - %v", container.ID, container.Labels["OBSRemotePort"])
			continue
		}
		vncPort, err := strconv.ParseInt(container.Labels["VNCDebugPort"], 10, 32)
		if err != nil {
			s.logger.Warnf("Failed to parse VNC debug port from running container %v - %v", container.ID, container.Labels["VNCDebugPort"])
			continue
		}

		dockerNetwork, err := strconv.ParseInt(container.Labels["DockerNetwork"], 10, 32)
		if err != nil {
			s.logger.Warnf("Failed to parse Docker network from running container %v - %v", container.ID, container.Labels["DockerNetwork"])
			continue
		}

		slot := Slot{
			TTYNum:        int32(ttyNum),
			OBSRemotePort: int32(obsPort),
			VNCDebugPort:  int32(vncPort),
			DockerNetwork: int32(dockerNetwork),
		}

		newInstance := &instance{
			containerID: container.ID,
			logger:      s.logger,
			docker:      s.docker,
			aws:         s.awsClient,
			slot:        slot,
		}

		s.instances[instanceID] = newInstance
		s.forceCreateSlot(instanceID, slot)
		s.wg.Add(1)

		defer func(inst *instance) {
			inst.startOBSConnection()
		}(newInstance)
	}

	return nil
}
