package moonlight

import (
	"context"
	"encoding/json"
	"fmt"
	"github.com/aws/aws-sdk-go/service/ec2"
	"net"
	"sync"
	"time"

	"code.justin.tv/amzn/TwirpGoLangAWSTransports/lambda"
	"code.justin.tv/event-engineering/acm-ca-go/pkg/certgen"
	"code.justin.tv/event-engineering/golibs/pkg/logging"
	db "code.justin.tv/event-engineering/moonlight-api/pkg/db"
	rpc "code.justin.tv/event-engineering/moonlight-api/pkg/rpc"
	controlRPC "code.justin.tv/event-engineering/moonlight-api/pkg/rpc/control"
	"code.justin.tv/event-engineering/moonlight-daemon/pkg/aws"
	awsBackend "code.justin.tv/event-engineering/moonlight-daemon/pkg/aws/backend"
	xorg "code.justin.tv/event-engineering/moonlight-daemon/pkg/xorg"
	"github.com/aws/aws-sdk-go/service/cloudwatch"
	dockerClient "github.com/fsouza/go-dockerclient"
	"github.com/golang/protobuf/ptypes"
)

var (
	healthCheckInterval      = 1 * time.Second
	unhealthyInstanceTimeout = 10 * time.Second
)

// Server is the interface that represents all Moonlight Server methods
type Server interface {
	Stop(stopInstances bool)
	Start()
}

type server struct {
	ID                         string
	logger                     logging.Logger
	moonlightAPI               controlRPC.InternalController
	wg                         sync.WaitGroup
	awsClient                  *aws.Client
	certGen                    certgen.Generator
	stopped                    bool
	instances                  map[string]Instance
	instanceLock               sync.Mutex
	listener                   net.Listener
	listenIP                   string
	tlsHost                    string
	configDone                 chan (interface{})
	docker                     *dockerClient.Client
	slotLock                   sync.Mutex
	maxSlots                   int
	slots                      map[string]Slot
	pciBusID                   string
	logDir                     string
	tmpDir                     string
	healthTicker               *time.Ticker
	moonlightBridgeNetworkName string
}

// New generates a new Moonlight Server instance
func New(internalLambdaARN, logDir, tmpDir string, ab awsBackend.Client, certGen certgen.Generator, logger logging.Logger) (Server, error) {
	listenIP := ""
	tlsHost := ""
	serverID := ""

	awsClient := aws.New(ab, logger)

	identity, err := awsClient.EC2MGetInstanceIdentityDocument()
	if err != nil {
		return nil, err
	}

	listenIP = identity.PrivateIP
	serverID = identity.InstanceID

	instance, err := awsClient.EC2DescribeInstances(&ec2.DescribeInstancesInput{
		InstanceIds: []*string{
			&identity.InstanceID,
		},
	})
	if err != nil {
		logger.Errorf("Failed to describe instances %v", err)
		return nil, err
	}

	if len(instance.Reservations) == 0 {
		logger.Errorf("Unexpected length of Reservations from describe instances %v", err)
		return nil, err
	}

	if len(instance.Reservations[0].Instances) == 0 {
		logger.Errorf("Unexpected length of Instances from describe instances %v", err)
		return nil, err
	}

	tlsHost = *instance.Reservations[0].Instances[0].PrivateDnsName

	docker, err := dockerClient.NewClientFromEnv()
	if err != nil {
		logger.Errorf("Failed to create docker client %v", err)
		return nil, err
	}

	pciBusID, err := xorg.GetNvidiaPCISlot()
	if err != nil {
		logger.Errorf("Failed to detect graphics card %v", err)
		return nil, err
	}

	logger.Debugf("Detected device with Bus ID %v", pciBusID)

	lambdaTransport := lambda.NewClient(awsClient.GetLambdaClient(), internalLambdaARN)

	s := &server{
		ID:                         serverID,
		logger:                     logger,
		moonlightAPI:               controlRPC.NewInternalControllerJSONClient("https://www.doesnt.matter.because.its.actually.lambda.transport.twitch.tv", lambdaTransport),
		awsClient:                  awsClient,
		certGen:                    certGen,
		instances:                  make(map[string]Instance),
		listenIP:                   listenIP,
		tlsHost:                    tlsHost,
		docker:                     docker,
		maxSlots:                   8,
		slots:                      make(map[string]Slot),
		pciBusID:                   pciBusID,
		logDir:                     logDir,
		tmpDir:                     tmpDir,
		moonlightBridgeNetworkName: "docker69",
	}

	err = s.configureNetwork()
	if err != nil {
		logger.Errorf("Failed to configure docker network - %v", err)
		return nil, err
	}

	return s, nil
}

func (s *server) Stop(stopInstances bool) {
	s.stopped = true
	s.configDone <- "lol"
	s.healthTicker.Stop()

	s.logger.Debug("Removing instances")
	s.instanceLock.Lock()
	for id := range s.instances {
		s.logger.Debugf("Removing instance with ID %v", id)
		s.removeInstance(id, stopInstances)
	}
	s.instanceLock.Unlock()

	s.logger.Debug("Stopping listener")
	s.stopListener()

	s.logger.Debug("Deregistering daemon")
	s.deregisterDaemon()

	s.logger.Debug("Done")
	s.wg.Done()
	s.wg.Wait()
}

func (s *server) Start() {
	// Set up OBS Cloud instances
	s.wg.Add(1)

	s.registerDaemon()

	s.populateRunningInstances()

	go s.updateOBSCloudConfig()

	go s.setUpHealthCheck()

	// Set up twirp listener for incoming commands
	go s.configureListener(s.listenIP, s.tlsHost)
}

func (s *server) configureNetwork() error {
	// Check network
	_, err := s.docker.NetworkInfo(s.moonlightBridgeNetworkName)

	if err != nil {
		// If the network doesn't exist, then create it
		if _, ok := err.(*dockerClient.NoSuchNetwork); ok {
			s.logger.Infof("Network %v doesn't exist, creating it", s.moonlightBridgeNetworkName)

			_, err = s.docker.CreateNetwork(dockerClient.CreateNetworkOptions{
				Driver: "bridge",
				Name:   s.moonlightBridgeNetworkName,
				IPAM: &dockerClient.IPAMOptions{
					Config: []dockerClient.IPAMConfig{
						dockerClient.IPAMConfig{
							Gateway: "172.18.1.1",
							Subnet:  "172.18.1.0/24",
						},
					},
				},
				Options: map[string]interface{}{
					"com.docker.network.bridge.enable_icc":           "false",
					"com.docker.network.bridge.enable_ip_masquerade": "true",
					"com.docker.network.driver.mtu":                  "1500",
					"com.docker.network.bridge.name":                 s.moonlightBridgeNetworkName,
				},
			})

			if err != nil {
				s.logger.Warnf("Failed to create network %v - %v", s.moonlightBridgeNetworkName, err)
				return err
			}
		} else {
			s.logger.Warnf("Error when checking network existence for %v - %v", s.moonlightBridgeNetworkName, err)
			return err
		}
	}

	return nil
}

func (s *server) setUpHealthCheck() {
	s.healthTicker = time.NewTicker(healthCheckInterval)
	for range s.healthTicker.C {
		statuses := make([]*controlRPC.SetInstanceStatusReq, 0)

		var numInstances int32

		// Check the health of all instances
		s.instanceLock.Lock()
		for id, instance := range s.instances {
			numInstances++

			// We don't want to do health checks for instances that haven't fully started yet
			if !instance.IsStarted() {
				continue
			}

			status := &controlRPC.SetInstanceStatusReq{
				InstanceId: id,
			}
			statuses = append(statuses, status)

			hb := instance.GetLastHeartbeat()

			lastStreamed, err := ptypes.TimestampProto(hb.LastStreamed)
			if err != nil {
				s.logger.Warnf("Failed to convert Time to protobuf timestamp %v", err)
			} else {
				status.LastStreamed = lastStreamed
			}

			if hb.LastUpdated.Add(unhealthyInstanceTimeout).Before(time.Now()) {
				status.Status = string(db.InstanceUnhealthy)
			} else {
				status.Status = string(db.InstanceHealthy)
			}

			// Temporary until we get real time websocket updates all the way back to the client
			status.RtmpMuted = hb.RTMPMuted

			heartbeatJSON, err := json.Marshal(hb.Heartbeat)
			if err != nil {
				s.logger.Warnf("Error marshalling heartbeat %v", err)
				status.HeartbeatJson = ""
			} else {
				status.HeartbeatJson = string(heartbeatJSON)
			}
		}
		s.instanceLock.Unlock()

		// Update the db with the latest status
		ctx := context.Background()
		_, err := s.moonlightAPI.BatchSetInstanceStatus(ctx, &controlRPC.BatchSetInstanceStatusReq{
			Statuses: statuses,
		})

		if err != nil {
			s.logger.Warnf("Failed to update instances status: %v", err)
		}

		resp, err := s.moonlightAPI.UpdateDaemonStatus(ctx, &controlRPC.UpdateDaemonStatusReq{
			ServerId:     s.ID,
			NumInstances: numInstances,
		})

		if err != nil {
			s.logger.Warnf("Failed to update daemon status: %v", err)
		}

		if !resp.Success {
			s.logger.Warnf("Failed to update daemon status: %v", resp.Message)
		}
	}
}

func (s *server) deregisterDaemon() {
	resp, err := s.moonlightAPI.DeregisterDaemon(context.Background(), &controlRPC.DeregisterDaemonReq{
		ServerId: s.ID,
	})

	if err == nil && resp.Success {
		return
	}

	var msg string

	if err != nil {
		msg = err.Error()
	} else {
		msg = resp.Message
	}

	s.logger.Warnf("Failed to deregister worker %s", msg)
}

func (s *server) registerDaemon() {
	for {
		if s.stopped {
			return
		}

		resp, err := s.moonlightAPI.RegisterDaemon(context.Background(), &controlRPC.RegisterDaemonReq{
			ServerId: s.ID,
			ApiUrl:   fmt.Sprintf("https://%s:%v", s.tlsHost, listenPort),
		})

		if err == nil && resp.Success {
			// TODO set up the config we get back from the registration call
			return
		}

		var msg string

		if err != nil {
			msg = err.Error()
		} else {
			msg = resp.Message
		}

		s.logger.Warnf("Failed to register worker, retrying in 5 seconds %s", msg)
		time.Sleep(5 * time.Second)
	}
}

func (s *server) updateOBSCloudConfig() {
	s.configDone = make(chan interface{})
	configTicker := time.NewTicker(10 * time.Second)

	for {
		if s.stopped {
			break
		}

		config, err := s.moonlightAPI.GetOBSCloudConfig(context.Background(), &controlRPC.GetOBSCloudConfigReq{
			ServerId: s.ID,
		})

		if err != nil {
			s.logger.Errorf("getting server config %v", err)
		} else {
			s.logger.Debugf("Got %v config entries", len(config.Instances))
			s.setupServer(config)
		}

		select {
		case <-configTicker.C:
			continue
		case <-s.configDone:
			break
		}
	}
}

func (s *server) setupServer(config *controlRPC.GetOBSCloudConfigResp) {
	if s.stopped {
		return
	}

	// Config is the source of truth, so add any instances that don't currently exist, and remove any we're currently connected to but aren't in the config
	for _, ci := range config.Instances {
		// This should never be true, but just in case the server returns data it shouldn't
		if ci.ServerId != s.ID {
			continue
		}

		if _, ok := s.instances[ci.Id]; !ok {
			// Add the instance
			s.instanceLock.Lock()
			s.addInstance(ci)
			s.instanceLock.Unlock()
		}
	}

	for id := range s.instances {
		if !instanceExistsInConfig(s.ID, id, config) {
			// Remove the instance
			s.instanceLock.Lock()
			s.removeInstance(id, true)
			s.instanceLock.Unlock()
		}
	}

	s.logger.Infof("Running instances: %v", len(s.instances))
	go s.putRunningInstances(len(s.instances))
}

func (s *server) putRunningInstances(instances int) {
	namespace := "MoonlightComposite"
	metricName := "RunningInstances"
	floatValue := float64(instances)
	instanceID := "InstanceId"
	now := time.Now()

	_, err := s.awsClient.CWPutMetricData(&cloudwatch.PutMetricDataInput{
		Namespace: &namespace,
		MetricData: []*cloudwatch.MetricDatum{
			&cloudwatch.MetricDatum{
				MetricName: &metricName,
				Value:      &floatValue,
				Timestamp:  &now,
				Dimensions: []*cloudwatch.Dimension{
					&cloudwatch.Dimension{
						Name:  &instanceID,
						Value: &s.ID,
					},
				},
			},
		},
	})

	if err != nil {
		s.logger.Warnf("Failed to put running instances %v", err)
	}
}

func instanceExistsInConfig(serverID, instanceID string, config *controlRPC.GetOBSCloudConfigResp) bool {
	for _, instance := range config.Instances {
		if instance.ServerId == serverID && instance.Id == instanceID {
			return true
		}
	}
	return false
}

func (s *server) addInstance(config *rpc.OBSCloudInstance) {
	if _, ok := s.instances[config.Id]; !ok {
		slotExists, slot := s.getSlot(config.Id)
		var err error

		if !slotExists {
			slot, err = s.allocateSlot(config.Id)
			if err != nil {
				s.logger.Warnf("Attempted to create instance from config but slot could not be allocated, %v", err)
				return
			}
		}

		instance := newInstance(slot, s.docker, s.awsClient, s.logDir, s.tmpDir, s.moonlightBridgeNetworkName, s.logger)
		s.instances[config.Id] = instance

		s.wg.Add(1)
		defer func(inst Instance, conf *rpc.OBSCloudInstance) {
			err := inst.Start(conf, s.pciBusID)
			status := db.InstanceRunning // I need to move these enum strings to somewhere better
			if err != nil {
				status = db.InstanceError
			}

			ctx := context.Background()
			_, err = s.moonlightAPI.SetInstanceStatus(ctx, &controlRPC.SetInstanceStatusReq{
				InstanceId: conf.Id,
				Status:     string(status),
			})

			if err != nil {
				s.logger.Warnf("Error updating instance [%v] status, %v", conf.Id, err)
			}
		}(instance, config)
	} else {
		s.logger.Warnf("Attempted to create instance with ID %v but it already existed", config.Id)
	}
}

func (s *server) removeInstance(id string, stopInstance bool) {
	if instance, ok := s.instances[id]; ok {
		s.logger.Debugf("Calling Stop on instance %v", id)
		err := instance.Stop(stopInstance)
		if err != nil {
			s.logger.Warnf("Failed to stop instance, %v", err)
			return
		}

		s.logger.Debugf("Deallocating slot %v", id)
		s.deallocateSlot(id)

		defer func() {
			s.wg.Done()
		}()

		s.logger.Debugf("Deleting instance %v", id)
		delete(s.instances, id)

		s.logger.Infof("Instance %v successfully removed", id)
	} else {
		s.logger.Warnf("Attempted to remove instance with ID %v but it didn't exist in s.instances", id)
	}
}
