package moonlight

import (
	"fmt"
)

type Slot struct {
	OBSRemotePort int32
	VNCDebugPort  int32
	DockerNetwork int32
	TTYNum        int32
}

const startOBSPort int32 = 4444
const startVNCPort int32 = 5901
const startDockerNetwork int32 = 1
const startTTYNum int32 = 11

func (s *server) getSlot(instanceID string) (bool, Slot) {
	s.slotLock.Lock()
	defer s.slotLock.Unlock()

	slot, ok := s.slots[instanceID]
	return ok, slot
}

func (s *server) deallocateSlot(instanceID string) error {
	s.slotLock.Lock()
	defer s.slotLock.Unlock()

	if _, ok := s.slots[instanceID]; !ok {
		return fmt.Errorf("Slot doesn't exist for this instance %v", instanceID)
	}

	delete(s.slots, instanceID)

	return nil
}

func (s *server) allocateSlot(instanceID string) (Slot, error) {
	s.slotLock.Lock()
	defer s.slotLock.Unlock()

	if len(s.slots) >= s.maxSlots {
		return Slot{}, fmt.Errorf("Server is full %v/%v", len(s.slots), s.maxSlots)
	}

	if _, ok := s.slots[instanceID]; ok {
		return Slot{}, fmt.Errorf("Instance already exists %v", instanceID)
	}

	slot := Slot{
		OBSRemotePort: s.getNextOBSPort(),
		VNCDebugPort:  s.getNextVNCPort(),
		DockerNetwork: s.getNextDockerNetwork(),
		TTYNum:        s.getNextTTYNum(),
	}

	s.slots[instanceID] = slot

	return slot, nil
}

func (s *server) forceCreateSlot(instanceID string, slot Slot) {
	s.slotLock.Lock()
	defer s.slotLock.Unlock()

	s.slots[instanceID] = slot
}

func (s *server) getNextOBSPort() int32 {
	current := startOBSPort
	for {
		portAllocated := false

		for _, slot := range s.slots {
			if slot.OBSRemotePort == current {
				portAllocated = true
				break
			}
		}

		if portAllocated {
			current++
		} else {
			return current
		}
	}
}

func (s *server) getNextVNCPort() int32 {
	current := startVNCPort
	for {
		portAllocated := false

		for _, slot := range s.slots {
			if slot.VNCDebugPort == current {
				portAllocated = true
				break
			}
		}

		if portAllocated {
			current++
		} else {
			return current
		}
	}
}

func (s *server) getNextTTYNum() int32 {
	current := startTTYNum
	for {
		ttyAllocated := false

		for _, slot := range s.slots {
			if slot.TTYNum == current {
				ttyAllocated = true
				break
			}
		}

		if ttyAllocated {
			current++
		} else {
			return current
		}
	}
}

func (s *server) getNextDockerNetwork() int32 {
	current := startDockerNetwork
	for {
		networkAllocated := false

		for _, slot := range s.slots {
			if slot.DockerNetwork == current {
				networkAllocated = true
				break
			}
		}

		if networkAllocated {
			current++
		} else {
			return current
		}
	}
}
