package obs

// Wrapper around obs-websocket-go to meet the CloudComposite interop functions
import (
	"code.justin.tv/event-engineering/golibs/pkg/logging"
	obsws "code.justin.tv/event-engineering/obs-websocket-go/pkg/obs"
)

type server struct {
	ws     obsws.Client
	logger logging.Logger
}

func NewRemoteClient(address string, port int, logger logging.Logger) Server {
	ws := obsws.NewClient(address, port, logger)

	return &server{
		ws:     ws,
		logger: logger,
	}
}

func (c *server) Connect() (chan error, error) {
	return c.ws.Go()
}

func (s *server) GetStatus() (*ServerStatus, error) {
	status, err := s.ws.GetStreamStatus()
	if err != nil {
		return nil, err
	}

	// TODO work on the rest of the data I want in here
	return &ServerStatus{
		IsStreaming: status.Streaming,
	}, nil
}

func (s *server) StartStream(streamKey string) error {
	err := s.ws.SetStreamKey(streamKey)
	if err != nil {
		return err
	}
	return s.ws.StartStream()
}

func (s *server) StopStream() error {
	err := s.ws.StopStream()
	if err != nil {
		return err
	}

	return s.ws.SetStreamKey("")
}

func (s *server) SwitchScene(sceneIdentifier string) error {
	return s.ws.SetCurrentScene(sceneIdentifier)
}

func (s *server) Close() error {
	return s.ws.Close()
}

func (s *server) GetSceneList() (*SceneList, error) {
	scenelist, err := s.ws.GetSceneList()
	if err != nil {
		return nil, err
	}

	resp := &SceneList{
		CurrentScene: scenelist.CurrentScene,
		Scenes:       make([]string, 0),
	}

	for _, scene := range scenelist.Scenes {
		resp.Scenes = append(resp.Scenes, scene.Name)
	}

	return resp, nil
}

func (s *server) Subscribe() chan obsws.Event {
	return s.ws.Subscribe()
}

func (s *server) SetHeartbeat(enable bool) error {
	err := s.ws.SetHeartbeat(enable)
	if err != nil {
		return err
	}
	return nil
}

func (s *server) MuteInput(source string, mute bool) error {
	err := s.ws.SetMute(source, mute)
	if err != nil {
		return err
	}
	return nil
}

func (s *server) GetMuteStatus(source string) (bool, error) {
	resp, err := s.ws.GetMute(source)
	if err != nil {
		return false, err
	}
	return resp.Muted, nil
}

func (s *server) UpdateOBSConfig(url string, width, height, x, y int) error {
	err := s.ws.SetBrowserSourceProperties(&obsws.SetBrowserSourcePropertiesRequest{
		Source: "OVERLAY",
		URL:    url,
		Width:  width,
		Height: height,
		FPS:    60,
	})
	if err != nil {
		return err
	}

	err = s.ws.SetSceneItemProperties(&obsws.SetSceneItemPropertiesRequest{
		SceneName: ".OVERLAY",
		Item:      "OVERLAY",
		Position: obsws.SceneItemPosition{
			X:         x,
			Y:         y,
			Alignment: obsws.AlignmentTop | obsws.AlignmentLeft,
		},
		Visible: true,
		Scale: obsws.SceneItemScale{
			X: 1,
			Y: 1,
		},
		Bounds: obsws.SceneItemBounds{
			Type: obsws.BoundsNone,
		},
	})

	return err
}
