package xorg

import (
	"code.justin.tv/event-engineering/moonlight-daemon/pkg/lspci"
	"fmt"
	"github.com/gobuffalo/packr"
	"os"
	"path"
	"strconv"
	"strings"
	"text/template"
)

type config struct {
	PCIBusID string
}

func GenerateXOrgConfig(outputDir, pciBusId string) error {
	box := packr.NewBox("./templates")

	templateString, err := box.FindString("xorg.conf")
	if err != nil {
		return err
	}

	tmpl, err := template.New("xorg.conf").Parse(templateString)
	if err != nil {
		return err
	}

	outputFile, err := os.Create(path.Join(outputDir, "xorg.conf"))
	if err != nil {
		return err
	}

	defer outputFile.Close()

	data := config{
		PCIBusID: pciBusId,
	}

	err = tmpl.Execute(outputFile, data)
	if err != nil {
		return err
	}

	return nil
}

func GetNvidiaPCISlot() (string, error) {
	l := lspci.New(false)

	if err := l.Run(); err != nil {
		return "", err
	}

	var slot string

	for pciAddress, data := range l.Data {
		// GM204GL [Tesla M60] is G3
		// Device 1eb8 is G4
		if data["Vendor"] == "NVIDIA Corporation" && (data["Device"] == "GM204GL [Tesla M60]" || data["Device"] == "Device 1eb8") {
			slot = pciAddress
			break
		}
	}

	if slot == "" {
		return "", fmt.Errorf("Could not find compatible device %v", l.Data)
	}

	return formatBusID(slot)
}

func formatBusID(input string) (string, error) {
	// 0000:00:1e.0
	parts := strings.Split(input, ":")
	parts2 := strings.Split(parts[2], ".")

	busID0, err := strconv.ParseInt(parts[1], 16, 64)
	if err != nil {
		return "", err
	}

	busID1, err := strconv.ParseInt(parts2[0], 16, 64)
	if err != nil {
		return "", err
	}

	busID2, err := strconv.ParseInt(parts2[1], 16, 64)
	if err != nil {
		return "", err
	}

	return fmt.Sprintf("PCI:%d:%d:%d", busID0, busID1, busID2), nil
}
