import cdk = require('@aws-cdk/core');
import ec2 = require('@aws-cdk/aws-ec2');
import lambda = require('@aws-cdk/aws-lambda');
import events = require('@aws-cdk/aws-events');
import eventTargets = require('@aws-cdk/aws-events-targets');
import les = require('@aws-cdk/aws-lambda-event-sources')
import sqs = require('@aws-cdk/aws-sqs')
import apig = require('@aws-cdk/aws-apigateway')
import iam = require('@aws-cdk/aws-iam')
import ssm = require('@aws-cdk/aws-ssm')
import ecr = require('@aws-cdk/aws-ecr')
import ddb = require('@aws-cdk/aws-dynamodb');

export interface APIEnvironment {
  TwitchClientID: string
  TwitchDevClientID: string
}

export interface DBEnvironment {
  InstancesTable: ddb.Table
  ServersTable: ddb.Table
  RTMPSourcesTable: ddb.Table
  UsersTable: ddb.Table
}

export interface APIProps {
  vpc: ec2.Vpc
  env: APIEnvironment
  db: DBEnvironment
  rtmpSourceURL: string
  ecrRepo: ecr.Repository
  createDevAPI: boolean
  bridgeLambdaRoleARN: string
  publicMediaURL: string
  daemonASGName: string
  daemonASGARN: string
  moonlightRootCAARN: string
  canAccessSystemBindleLockID: string
  adminBindleLockID: string
  opsBindleLockID: string
}

export class MoonlightAPI extends cdk.Construct {
  ControlLambda: lambda.Function
  AdminLambda: lambda.Function
  ExternalLambda: lambda.Function
  JobRunnerLambda: lambda.Function
  ControlLambdaARNSSMParameter: ssm.StringParameter

  constructor(scope: cdk.Construct, id: string, props: APIProps) {
    super(scope, id);

    const allowDDBAccess = new iam.PolicyStatement({
      effect: iam.Effect.ALLOW,
      actions: [
        "dynamodb:Query",
        "dynamodb:Scan",
        "dynamodb:GetItem",
        "dynamodb:PutItem",
        "dynamodb:UpdateItem",
        "dynamodb:DeleteItem",
      ],
      resources: [
        props.db.InstancesTable.tableArn,
        props.db.InstancesTable.tableArn + "/*",
        props.db.ServersTable.tableArn,
        props.db.ServersTable.tableArn + "/*",
        props.db.RTMPSourcesTable.tableArn,
        props.db.RTMPSourcesTable.tableArn + "/*",
        props.db.UsersTable.tableArn,
        props.db.UsersTable.tableArn + "/*",
      ]
    })

    const allowAutoscalingOperations = new iam.PolicyStatement({
      effect: iam.Effect.ALLOW,
      actions: [
        "autoscaling:SetDesiredCapacity",
        "autoscaling:TerminateInstanceInAutoScalingGroup",
      ],
      resources: [props.daemonASGARN]
    })

    const allowMoonlightRootCACertRetrieval = new iam.PolicyStatement({
      effect: iam.Effect.ALLOW,
      actions: [
        "acm-pca:GetCertificateAuthorityCertificate"
      ],
      resources: [props.moonlightRootCAARN]
    })

    // Job Runner
    this.JobRunnerLambda = new lambda.Function(this, 'MoonlightJobRunner', {
      vpc: props.vpc,
      runtime: lambda.Runtime.GO_1_X,
      memorySize: 512,
      timeout: cdk.Duration.seconds(60),
      handler: 'bin/linux/jobrunner/moonlight-jobrunner',
      code: lambda.Code.asset('build/api/src/code.justin.tv/event-engineering/moonlight-api/bin/linux/moonlight-jobrunner.zip'),
      environment: {
        "instancesTableName": props.db.InstancesTable.tableName,
        "serversTableName": props.db.ServersTable.tableName,
        "rtmpSourcesTableName": props.db.RTMPSourcesTable.tableName,
        "usersTableName": props.db.UsersTable.tableName,
        "daemonASGName": props.daemonASGName,
        "moonlightRootCAARN": props.moonlightRootCAARN,
      }
    })

    this.JobRunnerLambda.addToRolePolicy(allowDDBAccess)
    this.JobRunnerLambda.addToRolePolicy(allowAutoscalingOperations)
    this.JobRunnerLambda.addToRolePolicy(allowMoonlightRootCACertRetrieval)

    const jobRunnerSQS = new sqs.Queue(this, 'JobRunnerQueue', {
      visibilityTimeout: cdk.Duration.seconds(30),
      receiveMessageWaitTime: cdk.Duration.seconds(20),
    });

    this.JobRunnerLambda.addEventSource(new les.SqsEventSource(jobRunnerSQS, {
      batchSize: 1,
    }))

    // Control API
    this.ControlLambda = new lambda.Function(this, 'MoonlightInternalAPI', {
      vpc: props.vpc,
      runtime: lambda.Runtime.GO_1_X,
      memorySize: 512,
      timeout: cdk.Duration.seconds(30),
      handler: 'bin/linux/internal/moonlight-internal',
      code: lambda.Code.asset('build/api/src/code.justin.tv/event-engineering/moonlight-api/bin/linux/moonlight-internal.zip'),
      environment: {
        "instancesTableName": props.db.InstancesTable.tableName,
        "serversTableName": props.db.ServersTable.tableName,
        "rtmpSourcesTableName": props.db.RTMPSourcesTable.tableName,
        "usersTableName": props.db.UsersTable.tableName,
        "jobRunnerQueueURL": jobRunnerSQS.queueUrl,
        "rtmpSourceURL": props.rtmpSourceURL,
        "ecrRepositoryURL": props.ecrRepo.repositoryUri,
        "publicMediaURL": props.publicMediaURL,
        "moonlightRootCAARN": props.moonlightRootCAARN,
      }
    })

    this.ControlLambda.addToRolePolicy(allowDDBAccess)
    this.ControlLambda.addToRolePolicy(allowMoonlightRootCACertRetrieval)

    this.ControlLambda.addToRolePolicy(new iam.PolicyStatement({
      effect: iam.Effect.ALLOW,
      actions: ["sqs:SendMessage"],
      resources: [jobRunnerSQS.queueArn],
    }))

    // Create an SSM parameter for the internal Lambda ARN so that it can be used by the composite daemon and RTMP server
    this.ControlLambdaARNSSMParameter = new ssm.StringParameter(this, 'InternalLambdaARNParameter', {
      parameterName: "/moonlight/internalLambdaARN",
      stringValue: this.ControlLambda.functionArn,
    })

    // Admin API
    this.AdminLambda = new lambda.Function(this, 'MoonlightAdminAPI', {
      vpc: props.vpc,
      runtime: lambda.Runtime.GO_1_X,
      memorySize: 512,
      timeout: cdk.Duration.seconds(30),
      handler: 'bin/linux/admin/moonlight-admin',
      code: lambda.Code.asset('build/api/src/code.justin.tv/event-engineering/moonlight-api/bin/linux/moonlight-admin.zip'),
      environment: {
        "instancesTableName": props.db.InstancesTable.tableName,
        "serversTableName": props.db.ServersTable.tableName,
        "rtmpSourcesTableName": props.db.RTMPSourcesTable.tableName,
        "usersTableName": props.db.UsersTable.tableName,
        "jobRunnerQueueURL": jobRunnerSQS.queueUrl,
        "rtmpSourceURL": props.rtmpSourceURL,
        "ecrRepositoryURL": props.ecrRepo.repositoryUri,
        "publicMediaURL": props.publicMediaURL,
        "moonlightRootCAARN": props.moonlightRootCAARN,
        "canAccessSystemBindleLockID": props.canAccessSystemBindleLockID,
        "adminBindleLockID": props.adminBindleLockID,
        "opsBindleLockID": props.opsBindleLockID,
      }
    })

    this.AdminLambda.addToRolePolicy(new iam.PolicyStatement({
      effect: iam.Effect.ALLOW,
      actions: [
        "brassservice:IsAuthorized",
        "brassservice:BatchIsAuthorized",
      ],
      resources: [
        '*'
      ]
    }))

    this.AdminLambda.addToRolePolicy(allowDDBAccess)
    this.AdminLambda.addToRolePolicy(allowMoonlightRootCACertRetrieval)

    // Allow the bridge lambda function to call the moonlight admin api
    this.AdminLambda.grantInvoke(new iam.ArnPrincipal(props.bridgeLambdaRoleARN))

    // External API
    const externalAPI = new apig.RestApi(this, 'moonlight-external-api', {
      deployOptions: {

      },
      defaultIntegration: undefined,
      defaultMethodOptions: undefined,
    })

    const externalLambda = new lambda.Function(this, 'MoonlightExternalAPI', {
      vpc: props.vpc,
      runtime: lambda.Runtime.GO_1_X,
      memorySize: 512,
      timeout: cdk.Duration.seconds(15),
      handler: 'bin/linux/external/moonlight-external',
      code: lambda.Code.asset('build/api/src/code.justin.tv/event-engineering/moonlight-api/bin/linux/moonlight-external.zip'),
      environment: {
        "clientId": props.env.TwitchClientID,
        "internalLambdaARN": this.ControlLambda.functionArn,
      }
    })

    // Allow the external API lambda function to call the internal API lambda function
    this.ControlLambda.grantInvoke(externalLambda.grantPrincipal)

    const externalAuthoriser = new lambda.Function(this, 'MoonlightExternalAuthoriser', {
      vpc: props.vpc,
      runtime: lambda.Runtime.GO_1_X,
      memorySize: 512,
      timeout: cdk.Duration.seconds(5),
      handler: 'bin/linux/external/moonlight-twitch-authoriser',
      code: lambda.Code.asset('build/api/src/code.justin.tv/event-engineering/moonlight-api/bin/linux/moonlight-external-authoriser.zip'),
      environment: {
        "clientId": props.env.TwitchClientID,
      }
    });

    if (props.createDevAPI) {
      externalAuthoriser.addEnvironment('devClientId', props.env.TwitchDevClientID)
    }

    const role = new iam.Role(this, 'MoonlightExternalAuthoriserRole', {
      assumedBy: new iam.ServicePrincipal('apigateway.amazonaws.com'),
    });
    externalAuthoriser.grantInvoke(role);

    const authorizerUri = `arn:aws:apigateway:${cdk.Stack.of(this).region}:lambda:path/2015-03-31/functions/${
      externalAuthoriser.functionArn
      }/invocations`;

    const authorizer = new FuckYou(this, 'Authorizer', {
      authorizerUri,
      authorizerResultTtlInSeconds: 0,
      authorizerCredentials: role.roleArn,
      name: 'rest-api-authorizer',
      restApiId: externalAPI.restApiId,
      type: 'REQUEST',
    });

    const proxyAll = externalAPI.root.addProxy({
      defaultIntegration: new apig.LambdaIntegration(externalLambda),
      defaultMethodOptions: {
        operationName: "ANY",
        authorizationType: apig.AuthorizationType.CUSTOM,
        authorizer: authorizer,
      }
    })

    addCorsOptions(proxyAll)

    if (props.createDevAPI) {
      new apig.Stage(this, 'MoonlightExternalDevApiStage', {
        deployment: new apig.Deployment(this, 'MoonlightExternalDevDeployment', {
          api: externalAPI,
          description: 'dev'
        }),
        stageName: 'dev',
      })

      // Allow API gateway dev stage to invoke the lambda function
      externalLambda.addPermission('APIGatewayDevStageLambdaInvokePermission', {
        principal: new iam.ServicePrincipal('apigateway.amazonaws.com'),
        sourceArn: externalAPI.arnForExecuteApi(undefined, "/{proxy+}", "dev")
      })
    }

    // Set up cloudwatch events to trigger consistency checking
    new events.Rule(this, "ConsistencyCheckerCloudwatchEvent", {
      schedule: events.Schedule.rate(cdk.Duration.minutes(1)),
      targets: [
        new eventTargets.LambdaFunction(this.JobRunnerLambda, {
          event: events.RuleTargetInput.fromObject({
            source: "aws.events",
            detail: {
              job_type: "CHECK_CONSISTENCY"
            }
          })
        })
      ],
      enabled: true,
    })

    // Set up cloudwatch events to trigger autoscaling monitoring
    new events.Rule(this, "DaemonAutoScalingCheckerCloudwatchEvent", {
      schedule: events.Schedule.rate(cdk.Duration.minutes(1)),
      targets: [
        new eventTargets.LambdaFunction(this.JobRunnerLambda, {
          event: events.RuleTargetInput.fromObject({
            source: "aws.events",
            detail: {
              job_type: "CHECK_SCALING"
            }
          })
        })
      ],
      enabled: true,
    })
  }
}

class FuckYou extends apig.CfnAuthorizer {
  constructor(scope: cdk.Construct, id: string, props: apig.CfnAuthorizerProps) {
    super(scope, id, props);
  }

  get authorizerId(): string {
    return this.ref;
  }
}

export function addCorsOptions(apiResource: apig.Resource) {
  apiResource.addMethod('OPTIONS', new apig.MockIntegration({
    integrationResponses: [{
      statusCode: '200',
      responseParameters: {
        'method.response.header.Access-Control-Allow-Headers': "'Content-Type,Authorization'",
        'method.response.header.Access-Control-Allow-Origin': "'*'",
        'method.response.header.Access-Control-Allow-Credentials': "'false'",
        'method.response.header.Access-Control-Allow-Methods': "'OPTIONS,GET,PUT,POST,DELETE'",
      },
    }],
    passthroughBehavior: apig.PassthroughBehavior.NEVER,
    requestTemplates: {
      "application/json": "{\"statusCode\": 200}"
    },
  }), {
      authorizationType: apig.AuthorizationType.NONE, // We don't want to run the authorisation on OPTIONS requests specifically
      methodResponses: [{
        statusCode: '200',
        responseParameters: {
          'method.response.header.Access-Control-Allow-Headers': true,
          'method.response.header.Access-Control-Allow-Methods': true,
          'method.response.header.Access-Control-Allow-Credentials': true,
          'method.response.header.Access-Control-Allow-Origin': true,
        },
      }]
    })
}
