import cdk = require('@aws-cdk/core');
import autoscaling = require('@aws-cdk/aws-autoscaling');
import iam = require('@aws-cdk/aws-iam');
import s3 = require('@aws-cdk/aws-s3');
import codedeploy = require('@aws-cdk/aws-codedeploy');

export interface DaemonDeployProps {
  asg: autoscaling.AutoScalingGroup
  deployBucketName: string
}

export class MoonlightDaemonDeploy extends cdk.Construct {

  constructor(scope: cdk.Construct, id: string, props: DaemonDeployProps) {
    super(scope, id);

    const s3Bucket = new s3.Bucket(this, 'MoonlightDaemonDeploymentBucket', {
      bucketName: props.deployBucketName,
    })

    const iamRole = new iam.Role(this, 'MoonlightCompositeRole', {
      assumedBy: new iam.ServicePrincipal("codedeploy.amazonaws.com"),
      managedPolicies: [iam.ManagedPolicy.fromAwsManagedPolicyName("service-role/AWSCodeDeployRole")]
    })

    iamRole.addToPolicy(new iam.PolicyStatement({
      effect: iam.Effect.ALLOW,
      actions: [
        "s3:ListBucket",
        "s3:GetObject"
      ],
      resources: [
        s3Bucket.bucketArn,
        s3Bucket.arnForObjects("*")
      ]
    }))

    const daemonApp = new codedeploy.ServerApplication(this, 'DaemonCodeDeploy', {
      applicationName: 'MoonlightDaemonDeploy',
    });

    new codedeploy.ServerDeploymentGroup(this, 'DaemonCodeDeployDeploymentGroup', {
      application: daemonApp,
      deploymentGroupName: 'DaemonDeploymentGroup',
      autoScalingGroups: [props.asg],
      installAgent: true,
      role: iamRole,
    });
  }
}
