import cdk = require('@aws-cdk/core');
import autoscaling = require('@aws-cdk/aws-autoscaling');
import ec2 = require('@aws-cdk/aws-ec2');
import iam = require('@aws-cdk/aws-iam');
import ecr = require('@aws-cdk/aws-ecr');

export interface DaemonProps {
  ami: { [region: string]: string }
  vpc: ec2.Vpc
  keyPairName: string
  deploymentBucketName: string
  ecrRepository: ecr.Repository
  alternativeECRRepositoryARN?: string
}

export class MoonlightDaemon extends cdk.Construct {
  ASG: autoscaling.AutoScalingGroup
  IAMRole: iam.Role
  SecurityGroup: ec2.SecurityGroup

  constructor(scope: cdk.Construct, id: string, props: DaemonProps) {
    super(scope, id);

    this.SecurityGroup = new ec2.SecurityGroup(this, 'MoonlightCompositeSG', {
      vpc: props.vpc,
      description: "Access control for the Moonlight Composite component",
      allowAllOutbound: true,
    });

    this.IAMRole = new iam.Role(this, 'MoonlightCompositeRole', {
      assumedBy: new iam.ServicePrincipal("ec2.amazonaws.com"),
      managedPolicies: [iam.ManagedPolicy.fromAwsManagedPolicyName("service-role/AmazonEC2RoleforSSM")]
    })

    // Cloudwatch agent for logs and metrics
    this.IAMRole.addManagedPolicy(iam.ManagedPolicy.fromAwsManagedPolicyName("CloudWatchAgentServerPolicy"))

    const ecrStatement = new iam.PolicyStatement({
      effect: iam.Effect.ALLOW,
      actions: [
        "ecr:GetLifecyclePolicyPreview",
        "ecr:GetDownloadUrlForLayer",
        "ecr:BatchGetImage",
        "ecr:DescribeImages",
        "ecr:DescribeRepositories",
        "ecr:ListTagsForResource",
        "ecr:ListImages",
        "ecr:BatchCheckLayerAvailability",
        "ecr:GetRepositoryPolicy",
        "ecr:GetLifecyclePolicy"
      ],
      resources: [props.ecrRepository.repositoryArn]
    })

    if (props.alternativeECRRepositoryARN) {
      ecrStatement.addResources(props.alternativeECRRepositoryARN)
    }

    this.IAMRole.addToPolicy(ecrStatement)

    this.IAMRole.addToPolicy(new iam.PolicyStatement({
      effect: iam.Effect.ALLOW,
      actions: ["ecr:GetAuthorizationToken"],
      resources: ["*"],
    }))

    this.IAMRole.addToPolicy(new iam.PolicyStatement({
      effect: iam.Effect.ALLOW,
      actions: ["ec2:DescribeInstances"],
      resources: ["*"],
    }))

    this.IAMRole.addToPolicy(new iam.PolicyStatement({
      effect: iam.Effect.ALLOW,
      actions: [
        "s3:Get*",
        "s3:List*"
      ],
      resources: [
        "arn:aws:s3:::" + props.deploymentBucketName,
        "arn:aws:s3:::" + props.deploymentBucketName + "/*"
      ]
    }))

    this.ASG = new autoscaling.AutoScalingGroup(this, 'ASG', {
      vpc: props.vpc,
      instanceType: new ec2.InstanceType('g4dn.4xlarge'),
      machineImage: new ec2.GenericLinuxImage(props.ami),
      keyName: props.keyPairName,
      vpcSubnets: {
        subnetType: ec2.SubnetType.PRIVATE,
      },
      minCapacity: 1,
      maxCapacity: 2,
      role: this.IAMRole,
    });

    this.ASG.addSecurityGroup(this.SecurityGroup)
  }
}
