import cdk = require('@aws-cdk/core');
import ec2 = require('@aws-cdk/aws-ec2');
import daemon = require('./moonlight-daemon/moonlight-daemon')
import daemonDeploy = require('./moonlight-daemon-deploy/moonlight-daemon-deploy')
import rtmpDeploy = require('./moonlight-rtmp-deploy/moonlight-rtmp-deploy')
import moonlightContainerReg = require('./moonlight-container-registry/moonlight-container-registry')
import rtmp = require('./moonlight-rtmp/moonlight-rtmp')
import api = require('./moonlight-api/moonlight-api')
import db = require('./moonlight-db/moonlight-db')
import iam = require('@aws-cdk/aws-iam');
import ssm = require('@aws-cdk/aws-ssm')

export interface StackProps {
  cdkProps: cdk.StackProps
  daemonAMI: { [region: string]: string }
  rtmpAMI: { [region: string]: string }
  apiEnv: api.APIEnvironment
  daemonDeployBucket: string
  rtmpDeployBucket: string
  daemonKeyPairName: string
  rtmpKeyPairName: string
  publicMediaURL: string
  createDevExternalAPI: boolean
  bridgeLambdaRoleARN: string
  moonlightRootCAARN: string
  secondaryECRIAMRoleARN?: string
  alternativeECRRepositoryARN?: string
  hostedZoneName: string
  rtmpHostName: string
  canAccessSystemBindleLockID: string
  adminBindleLockID: string
  opsBindleLockID: string
}

export class MoonlightInfraStack extends cdk.Stack {
  constructor(scope: cdk.Construct, id: string, props: StackProps) {
    super(scope, id, props.cdkProps);

    // The code that defines your stack goes here
    const vpc = new ec2.Vpc(this, 'MoonlightVpc', {
      maxAzs: 3,
      cidr: '172.20.0.0/16',
    });

    const ecrRepo = new moonlightContainerReg.ContainerRepository(this, 'ECRRepository', {
      secondaryIAMRoleARN: props.secondaryECRIAMRoleARN,
    })

    const moonlightDB = new db.MoonlightDb(this, 'MoonlightDb')

    const composite = new daemon.MoonlightDaemon(this, 'MoonlightDaemon', {
      vpc,
      ami: props.daemonAMI,
      keyPairName: props.daemonKeyPairName,
      deploymentBucketName: props.daemonDeployBucket,
      ecrRepository: ecrRepo.Repository,
      alternativeECRRepositoryARN: props.alternativeECRRepositoryARN,
    })

    const rtmpComponent = new rtmp.MoonlightRTMP(this, 'MoonlightRTMP', {
      vpc,
      ami: props.rtmpAMI,
      keyPairName: props.rtmpKeyPairName,
      deploymentBucketName: props.rtmpDeployBucket,
      moonlightCompositeSG: composite.SecurityGroup,
      hostedZoneName: props.hostedZoneName,
      rtmpHostName: props.rtmpHostName,
    })

    const moonlightAPI = new api.MoonlightAPI(this, 'MoonlightAPI', {
      vpc,
      env: props.apiEnv,
      createDevAPI: props.createDevExternalAPI,
      bridgeLambdaRoleARN: props.bridgeLambdaRoleARN,
      db: {
        InstancesTable: moonlightDB.InstancesTable,
        ServersTable: moonlightDB.ServersTable,
        RTMPSourcesTable: moonlightDB.RTMPSourcesTable,
        UsersTable: moonlightDB.UsersTable,
      },
      rtmpSourceURL: rtmpComponent.PrivateNLB.loadBalancerDnsName,
      daemonASGName: composite.ASG.autoScalingGroupName,
      daemonASGARN: composite.ASG.autoScalingGroupArn,
      publicMediaURL: props.publicMediaURL,
      ecrRepo: ecrRepo.Repository,
      moonlightRootCAARN: props.moonlightRootCAARN,
      canAccessSystemBindleLockID: props.canAccessSystemBindleLockID,
      adminBindleLockID: props.adminBindleLockID,
      opsBindleLockID: props.opsBindleLockID,
    })

    // Allow the daemon and RTMP server to call the internal API
    moonlightAPI.ControlLambda.grantInvoke(composite.IAMRole)
    moonlightAPI.ControlLambda.grantInvoke(rtmpComponent.IAMRole)

    // Create an SSM parameter for the root CA ARN so that it can be used by the composite daemon and RTMP server
    const rootCAARNParameter = new ssm.StringParameter(this, 'RootCAARNParameter', {
      parameterName: "/moonlight/moonlightRootCAARN",
      stringValue: props.moonlightRootCAARN,
    })

    // This SSM stuff has to be done here otherwise we have components referencing each other
    const ssmAccess = new iam.PolicyStatement({
      effect: iam.Effect.ALLOW,
      actions: [
        "ssm:GetParameter",
        "ssm:GetParameters",
      ],
      resources: [
        moonlightAPI.ControlLambdaARNSSMParameter.parameterArn,
        rootCAARNParameter.parameterArn,
      ]
    })

    // Allow composite to generate TLS certs using the Moonlight root CA
    const acmAccess = new iam.PolicyStatement({
      effect: iam.Effect.ALLOW,
      actions: [
        "acm:ExportCertificate",
        "acm:DescribeCertificate",
        "acm:RequestCertificate",
        "acm:GetCertificate",
        "acm:ListCertificates",
        "acm:ListTagsForCertificate",
        "acm-pca:IssueCertificate",
        "acm-pca:GetCertificate",
      ],
      resources: ["*"]
    })

    rtmpComponent.IAMRole.addToPolicy(ssmAccess)
    composite.IAMRole.addToPolicy(ssmAccess)
    composite.IAMRole.addToPolicy(acmAccess)

    new daemonDeploy.MoonlightDaemonDeploy(this, 'MoonlightDaemonDeploy', {
      asg: composite.ASG,
      deployBucketName: props.daemonDeployBucket,
    })

    new rtmpDeploy.MoonlightRtmpDeploy(this, 'MoonlightRtmpDeploy', {
      asg: rtmpComponent.ASG,
      deployBucketName: props.rtmpDeployBucket,
    })

    // Allow the API to talk directly to the composite instances
    composite.ASG.connections.allowFrom(moonlightAPI.ControlLambda, ec2.Port.tcp(8396), 'Allow API communication from control api')
    composite.ASG.connections.allowFrom(moonlightAPI.AdminLambda, ec2.Port.tcp(8396), 'Allow API communication from admin api')
    composite.ASG.connections.allowFrom(moonlightAPI.JobRunnerLambda, ec2.Port.tcp(8396), 'Allow API communication from job runner')
  }
}
