import cdk = require('@aws-cdk/core');
import autoscaling = require('@aws-cdk/aws-autoscaling');
import iam = require('@aws-cdk/aws-iam');
import s3 = require('@aws-cdk/aws-s3');
import codedeploy = require('@aws-cdk/aws-codedeploy');

export interface RtmpDeployProps {
  asg: autoscaling.AutoScalingGroup
  deployBucketName: string
}

export class MoonlightRtmpDeploy extends cdk.Construct {

  constructor(scope: cdk.Construct, id: string, props: RtmpDeployProps) {
    super(scope, id);

    const s3Bucket = new s3.Bucket(this, 'MoonlightRtmpDeploymentBucket', {
      bucketName: props.deployBucketName,
    })

    const iamRole = new iam.Role(this, 'MoonlightRtmpRole', {
      assumedBy: new iam.ServicePrincipal("codedeploy.amazonaws.com"),
      managedPolicies: [iam.ManagedPolicy.fromAwsManagedPolicyName("service-role/AWSCodeDeployRole")]
    })

    iamRole.addToPolicy(new iam.PolicyStatement({
      effect: iam.Effect.ALLOW,
      actions: [
        "s3:ListBucket",
        "s3:GetObject"
      ],
      resources: [
        s3Bucket.bucketArn,
        s3Bucket.arnForObjects("*")
      ]
    }))

    const rtmpApp = new codedeploy.ServerApplication(this, 'RtmpCodeDeploy', {
      applicationName: 'MoonlightRtmpDeploy',
    });

    new codedeploy.ServerDeploymentGroup(this, 'RtmpCodeDeployDeploymentGroup', {
      application: rtmpApp,
      deploymentGroupName: 'RtmpDeploymentGroup',
      autoScalingGroups: [props.asg],
      installAgent: true,
      role: iamRole,
    });
  }
}
