import cdk = require('@aws-cdk/core');
import autoscaling = require('@aws-cdk/aws-autoscaling');
import ec2 = require('@aws-cdk/aws-ec2');
import elb = require('@aws-cdk/aws-elasticloadbalancingv2');
import iam = require('@aws-cdk/aws-iam');
import route53 = require('@aws-cdk/aws-route53');
import route53targets = require('@aws-cdk/aws-route53-targets');
import acm = require('@aws-cdk/aws-certificatemanager');

export interface RTMPProps {
  ami: { [region: string]: string }
  vpc: ec2.Vpc
  moonlightCompositeSG: ec2.SecurityGroup
  deploymentBucketName: string
  keyPairName: string
  hostedZoneName: string
  rtmpHostName: string
}

export class MoonlightRTMP extends cdk.Construct {
  ASG: autoscaling.AutoScalingGroup
  IAMRole: iam.Role
  PublicNLB: elb.NetworkLoadBalancer
  PrivateNLB: elb.NetworkLoadBalancer
  SecurityGroup: ec2.SecurityGroup

  constructor(scope: cdk.Construct, id: string, props: RTMPProps) {
    super(scope, id);

    const hostedZone = route53.HostedZone.fromLookup(this, 'MoonlightHostedZoneRef', {
      domainName: props.hostedZoneName,
    })

    this.SecurityGroup = new ec2.SecurityGroup(this, 'MoonlightRTMPSG', {
      vpc: props.vpc,
      description: "Access control for the Moonlight RTMP component",
      allowAllOutbound: true,
    });

    this.SecurityGroup.addIngressRule(ec2.Peer.anyIpv4(), ec2.Port.tcp(1935), 'Allow RTMP from anywhere');
    this.SecurityGroup.addIngressRule(ec2.Peer.anyIpv4(), ec2.Port.tcp(443), 'Allow RTMPS from anywhere');
    this.SecurityGroup.addIngressRule(ec2.Peer.ipv4(props.vpc.vpcCidrBlock), ec2.Port.allTcp(), 'Allow any port internally'); // TODO I can't get security group references to work here, I think it's because it's going through NLB
    this.SecurityGroup.addIngressRule(ec2.Peer.ipv4(props.vpc.vpcCidrBlock), ec2.Port.tcp(1936), 'Allow internal connections on 1936 from anywhere in the VPC');

    this.IAMRole = new iam.Role(this, 'MoonlightRtmpRole', {
      assumedBy: new iam.ServicePrincipal("ec2.amazonaws.com"),
      managedPolicies: [iam.ManagedPolicy.fromAwsManagedPolicyName("service-role/AmazonEC2RoleforSSM")]
    })

    // Cloudwatch agent for logs and metrics
    this.IAMRole.addManagedPolicy(iam.ManagedPolicy.fromAwsManagedPolicyName("CloudWatchAgentServerPolicy"))

    this.IAMRole.addToPolicy(new iam.PolicyStatement({
      effect: iam.Effect.ALLOW,
      actions: [
        "s3:Get*",
        "s3:List*",
      ],
      resources: [
        "arn:aws:s3:::" + props.deploymentBucketName,
        "arn:aws:s3:::" + props.deploymentBucketName + "/*"
      ]
    }))

    this.ASG = new autoscaling.AutoScalingGroup(this, 'ASG', {
      vpc: props.vpc,
      instanceType: ec2.InstanceType.of(ec2.InstanceClass.T3, ec2.InstanceSize.MEDIUM),
      machineImage: new ec2.GenericLinuxImage(props.ami),
      keyName: props.keyPairName,
      vpcSubnets: {
        subnetType: ec2.SubnetType.PRIVATE,
      },
      minCapacity: 1,
      maxCapacity: 1,
      role: this.IAMRole,
    });

    this.ASG.addSecurityGroup(this.SecurityGroup)

    this.PublicNLB = new elb.NetworkLoadBalancer(this, 'ExternalNLB', {
      vpc: props.vpc,
      internetFacing: true,
      loadBalancerName: 'MoonlightRtmpExternalNLB',
    })

    const externalTargetGroup = new elb.NetworkTargetGroup(this, 'MoonlightRtmpExternalTargetGroup', {
      vpc: props.vpc,
      port: 1935,
    })

    this.ASG.attachToNetworkTargetGroup(externalTargetGroup)

    this.PublicNLB.addListener('MoonlightRTMPExternalListener', {
      port: 1935,
      defaultTargetGroups: [externalTargetGroup],
    })

    const rtmpsCert = new acm.DnsValidatedCertificate(this, 'MoonlightRtmpCertificate', {
      domainName: props.rtmpHostName,
      hostedZone: hostedZone,
    });

    this.PublicNLB.addListener('MoonlightRTMPSExternalListener', {
      port: 443,
      protocol: elb.Protocol.TLS,
      certificates: [
        elb.ListenerCertificate.fromCertificateManager(rtmpsCert)
      ],
      defaultTargetGroups: [externalTargetGroup],
    })

    // Create DNS entries for the public RTMP NLB
    new route53.ARecord(this, 'MoonlightRtmpARecord', {
      zone: hostedZone,
      recordName: props.rtmpHostName,
      target: route53.RecordTarget.fromAlias(new route53targets.LoadBalancerTarget(this.PublicNLB))
    })

    this.PrivateNLB = new elb.NetworkLoadBalancer(this, 'InternalNLB', {
      vpc: props.vpc,
      internetFacing: false,
      loadBalancerName: 'MoonlightRtmpInternalNLB',
    })

    const internalTargetGroup = new elb.NetworkTargetGroup(this, 'MoonlightRtmpInternalTargetGroup', {
      vpc: props.vpc,
      port: 1936,
    })

    this.ASG.attachToNetworkTargetGroup(internalTargetGroup)

    this.PrivateNLB.addListener('MoonlightRTMPInternalListener', {
      port: 1935,
      defaultTargetGroups: [internalTargetGroup],
    })
  }
}
