package main

import (
	"flag"
	"fmt"
	"io"
	"net"
	"os"
	"os/signal"
	"path"
	"syscall"
	"time"

	awsBackend "code.justin.tv/event-engineering/moonlight-rtmp/pkg/aws/backend"
	"code.justin.tv/event-engineering/moonlight-rtmp/pkg/router"
	"code.justin.tv/event-engineering/moonlight-rtmp/pkg/util"
	rtmplog "code.justin.tv/video/gortmp/pkg/log"
	"github.com/aws/aws-sdk-go/service/ssm"
	"github.com/sirupsen/logrus"
)

const (
	constConnTimeout = 60 * time.Second
)

var (
	logRoot = flag.String("logdir", "/var/opt/moonlight-rtmp/log", "Log directory location")
	logger  logrus.FieldLogger
)

func main() {
	flag.Parse()

	baseLogger := logrus.New()
	baseLogger.SetLevel(logrus.InfoLevel)
	baseLogger.SetNoLock()
	baseLogger.SetFormatter(&logrus.JSONFormatter{})
	logger = baseLogger.WithField("pid", os.Getpid())

	logLevel := rtmplog.ParseLogLevel("debug")
	rtmplog.SetLogLevel(logLevel)

	err := os.MkdirAll(*logRoot, os.ModeDir)
	if err != nil {
		panic(err)
	}

	logFile, err := os.OpenFile(path.Join(*logRoot, "moonlight-rtmp.log"), os.O_APPEND|os.O_WRONLY|os.O_CREATE, 0766)
	if err != nil {
		panic(err)
	}
	defer logFile.Close()

	mw := io.MultiWriter(os.Stdout, logFile)
	baseLogger.SetOutput(mw)

	ab, err := awsBackend.New("us-west-2")
	if err != nil {
		panic(err)
	}

	var internalLambdaARN string
	internalLambdaARNParamName := "/moonlight/internalLambdaARN"

	params, err := ab.SSMGetParameters(&ssm.GetParametersInput{
		Names: []*string{
			&internalLambdaARNParamName,
		},
	})

	if err != nil {
		panic(err)
	}

	for _, param := range params.Parameters {
		switch *param.Name {
		case internalLambdaARNParamName:
			internalLambdaARN = *param.Value
			break
		}
	}

	conn, err := getConnFromFD()
	if err != nil {
		logger.Warnf("Failed to get net.Conn from fd with: %s\n", err)
		os.Exit(1)
	}

	errCh := make(chan error)

	router.Create(conn, internalLambdaARN, ab, errCh, logger)

	osSig := make(chan os.Signal, 1)
	signal.Notify(osSig, os.Interrupt, syscall.SIGTERM, syscall.SIGQUIT)

	select {
	case sig := <-osSig:
		logger.Infof("Received signal %v\n", sig)
		conn.Close()
		return
	case <-errCh:
		return
	}
}

func getConnFromFD() (net.Conn, error) {
	file := os.NewFile(3, "")
	defer file.Close()
	if err := syscall.SetNonblock(int(file.Fd()), true); err != nil {
		return nil, err
	}
	conn, err := net.FileConn(file)
	if err != nil {
		return nil, err
	}
	tcpConn, ok := conn.(*net.TCPConn)
	if !ok {
		conn.Close()
		return nil, fmt.Errorf("Cannot get TCPConn from file")
	}
	return &util.TimeoutConn{FDConn: tcpConn, Timeout: constConnTimeout}, nil
}
