package main

import (
	"flag"
	"fmt"
	"net"
	"net/http"
	"os"
	"os/signal"
	"path"
	"sync"
	"syscall"
	"time"

	goctx "context"

	"io"

	awsBackend "code.justin.tv/event-engineering/moonlight-rtmp/pkg/aws/backend"
	"code.justin.tv/event-engineering/moonlight-rtmp/pkg/health"
	"code.justin.tv/event-engineering/moonlight-rtmp/pkg/util"
	rtmplog "code.justin.tv/video/gortmp/pkg/log"
	"code.justin.tv/video/gortmp/pkg/rtmp"
	"github.com/sirupsen/logrus"
)

// note that all flags/args provided to goingest will be passed along to goingestworker
var (
	logRoot = flag.String("logdir", "/var/opt/moonlight-rtmp/log", "Log directory location")
	logger  logrus.FieldLogger
)

type ProcessType int32

const (
	ProcessTypeInternal ProcessType = 0
	ProcessTypeExternal ProcessType = 1
)

type ProcessSpawner struct {
	processType ProcessType
	logger      logrus.FieldLogger
}

func main() {
	flag.Parse()

	baseLogger := logrus.New()
	baseLogger.SetLevel(logrus.InfoLevel)
	baseLogger.SetNoLock()
	baseLogger.SetFormatter(&logrus.JSONFormatter{})

	logLevel := rtmplog.ParseLogLevel("info")
	rtmplog.SetLogLevel(logLevel)

	logger = baseLogger.WithField("pid", os.Getpid())

	err := os.MkdirAll(*logRoot, os.ModeDir)
	if err != nil {
		panic(err)
	}

	logFile, err := os.OpenFile(path.Join(*logRoot, "moonlight-rtmp.log"), os.O_APPEND|os.O_WRONLY|os.O_CREATE, 0766)
	if err != nil {
		panic(err)
	}
	defer logFile.Close()

	mw := io.MultiWriter(os.Stdout, logFile)
	baseLogger.SetOutput(mw)

	ab, err := awsBackend.New("us-west-2")
	if err != nil {
		panic(err)
	}

	// We're gonna listen on 1935 for external incoming RTMP connections and 1936 for internal connections from the composite instances
	// For now we're just gonna use security groups to prevent people externally accessing port 1936 but eventually it's probably better
	// to make the internal listen on the internal IP and the external on the external and use security groups aswell
	var wg sync.WaitGroup
	hup := make(chan os.Signal, 1)

	instanceMetadata, err := ab.EC2MGetInstanceIdentityDocument()
	if err != nil {
		panic(err)
	}

	extLn, err := net.Listen("tcp", net.JoinHostPort(instanceMetadata.PrivateIP, "1935"))
	if err != nil {
		panic(fmt.Errorf("Could not create listener: %v", err))
	}
	defer extLn.Close()

	externalMultiLn := util.NewMultiListener(extLn, 10*time.Second)

	externalHealthCheck, err := health.NewHealthCheckHandler()
	if err != nil {
		panic(fmt.Errorf("external health check init failed: %s", err))
	}

	// external http listener
	go func() {
		err := http.Serve(externalMultiLn.HttpListener(), externalHealthCheck)
		logger.Println("External HTTP listener ended with:", err.Error())
		hup <- nil // this is fatal, so clean up and exit
	}()

	externalServer := rtmp.NewServer(rtmp.ServerConfig{
		Handler: &ProcessSpawner{
			processType: ProcessTypeExternal,
			logger:      logger,
		},
	})

	go func() {
		svErr := externalServer.ServeWithWaitGroup(externalMultiLn.RtmpListener(), &wg)
		logger.Infof("Done listening on %s: %s", externalMultiLn.RtmpListener().Addr(), svErr)
	}()

	intLn, err := net.Listen("tcp", net.JoinHostPort(instanceMetadata.PrivateIP, "1936"))
	if err != nil {
		panic(fmt.Errorf("Could not create listener: %v", err))
	}
	defer intLn.Close()

	internalMultiLn := util.NewMultiListener(intLn, 10*time.Second)

	internalHealthCheck, err := health.NewHealthCheckHandler()
	if err != nil {
		panic(fmt.Errorf("internal health check init failed: %s", err))
	}

	// internal http listener
	go func() {
		err := http.Serve(internalMultiLn.HttpListener(), internalHealthCheck)
		logger.Println("Internal HTTP listener ended with:", err.Error())
		hup <- nil // this is fatal, so clean up and exit
	}()

	internalServer := rtmp.NewServer(rtmp.ServerConfig{
		Handler: &ProcessSpawner{
			processType: ProcessTypeInternal,
			logger:      logger,
		},
	})

	go func() {
		err := internalServer.ServeWithWaitGroup(internalMultiLn.RtmpListener(), &wg)
		logger.Infof("Done listening on %s: %s", internalMultiLn.RtmpListener().Addr(), err)
	}()

	// receive one sighup
	signal.Notify(hup, syscall.SIGHUP)

	logger.Info("Server started listening")

	select {
	case <-hup:
		// don't catch any more signals
		signal.Stop(hup)

		// close our listen socket
		// SetDeadline helps make sure the socket actually gets closed right away
		extLn.Close()
		if tcpLn, ok := extLn.(*net.TCPListener); ok {
			tcpLn.SetDeadline(time.Now())
		}

		wg.Wait()
		logger.Printf("Done serving")
		return
	}
}

func (ps *ProcessSpawner) ServeRTMP(ctx goctx.Context, conn net.Conn) error {
	ps.logger.Infof("Got new connection from %v", conn.RemoteAddr())

	fdConn, ok := conn.(util.FDConn)
	if !ok {
		return fmt.Errorf("unable to assert type for conn")
	}
	fd, err := fdConn.File()
	if err != nil {
		return err
	}
	readEnd, _, err := os.Pipe()
	if err != nil {
		return err
	}

	attrs := &os.ProcAttr{Files: []*os.File{readEnd,
		os.Stdout,
		os.Stderr,
		fd},
	}

	processName := "/opt/moonlight-rtmp/moonlight-rtmp-internalworker-current"
	if ps.processType == ProcessTypeExternal {
		processName = "/opt/moonlight-rtmp/moonlight-rtmpworker-current"
	}

	procArgs := []string{processName}
	procArgs = append(procArgs, os.Args[1:]...)

	p, err := os.StartProcess(processName, procArgs, attrs)
	if err != nil {
		return err
	}
	logger.Infof("New process created with PID: %d", p.Pid)
	conn.Close()

	go func() {
		pstate, err := p.Wait()
		logger.Infof("Process %d exited with state: %s", p.Pid, pstate)
		if err != nil {
			logger.Infof("Wait returned error: %s", err)
		}
		time.Sleep(time.Second)
		fd.Close()
	}()

	return nil
}
