package main

import (
	"flag"
	"fmt"
	"io"
	"net"
	"os"
	"os/signal"
	"path"
	"syscall"
	"time"

	evrtmplog "code.justin.tv/event-engineering/gortmp/pkg/log" // this is messy while we're half in video/gortmp and half in event-engineering/gortmp
	internal "code.justin.tv/event-engineering/moonlight-api/pkg/rpc"
	awsBackend "code.justin.tv/event-engineering/moonlight-rtmp/pkg/aws/backend"
	"code.justin.tv/event-engineering/moonlight-rtmp/pkg/director"
	"code.justin.tv/event-engineering/moonlight-rtmp/pkg/util"
	rtmplog "code.justin.tv/video/gortmp/pkg/log"
	"github.com/aws/aws-sdk-go/service/ssm"
	"github.com/sirupsen/logrus"
)

const (
	constConnTimeout = 60 * time.Second
)

var (
	logRoot = flag.String("logdir", "/var/opt/moonlight-rtmp/log", "Log directory location")
	logger  logrus.FieldLogger
)

func main() {
	flag.Parse()

	baseLogger := logrus.New()
	baseLogger.SetLevel(logrus.DebugLevel)
	baseLogger.SetNoLock()
	baseLogger.SetFormatter(&logrus.JSONFormatter{})
	logger = baseLogger.WithField("pid", os.Getpid())

	rtmplog.SetLogLevel(rtmplog.ParseLogLevel("info"))
	evrtmplog.SetLogLevel(evrtmplog.ParseLogLevel("info"))

	err := os.MkdirAll(*logRoot, os.ModeDir)
	if err != nil {
		panic(err)
	}

	logFile, err := os.OpenFile(path.Join(*logRoot, "moonlight-rtmp.log"), os.O_APPEND|os.O_WRONLY|os.O_CREATE, 0766)
	if err != nil {
		panic(err)
	}
	defer logFile.Close()

	mw := io.MultiWriter(os.Stdout, logFile)
	baseLogger.SetOutput(mw)

	ab, err := awsBackend.New("us-west-2")
	if err != nil {
		panic(err)
	}

	instanceMetadata, err := ab.EC2MGetInstanceIdentityDocument()
	if err != nil {
		panic(err)
	}

	conn, err := getConnFromFD()
	if err != nil {
		logger.Warnf("Failed to get net.Conn from fd with: %s\n", err)
		os.Exit(1)
	}

	var internalLambdaARN string
	internalLambdaARNParamName := "/moonlight/internalLambdaARN"

	params, err := ab.SSMGetParameters(&ssm.GetParametersInput{
		Names: []*string{
			&internalLambdaARNParamName,
		},
	})

	if err != nil {
		panic(err)
	}

	for _, param := range params.Parameters {
		switch *param.Name {
		case internalLambdaARNParamName:
			internalLambdaARN = *param.Value
			break
		}
	}

	server, err := director.NewServer(conn, instanceMetadata.PrivateIP, internalLambdaARN, ab, logger)

	if err != nil {
		panic(err)
	}

	server.Start()

	events := server.Subscribe()

	go func() {
		for {
			event := <-events
			switch event.Type {
			case internal.RTMPEvent_StreamStarted:
				logger.Infof("Stream %v started", event.StreamID)
				break
			case internal.RTMPEvent_StreamDisconnected:
				logger.Infof("Stream %v disconnected", event.StreamID)
				break
			case internal.RTMPEvent_StreamStarvationStart:
				logger.Infof("Stream %v is starved", event.StreamID)
				break
			case internal.RTMPEvent_StreamStarvationEnd:
				logger.Infof("Stream %v is no longer starved", event.StreamID)
				break
			case internal.RTMPEvent_StreamBitrateLow:
				logger.Infof("Stream %v bitrate is too low", event.StreamID)
				break
			case internal.RTMPEvent_StreamBitrateOK:
				logger.Infof("Stream %v bitrate is ok", event.StreamID)
				break
			}
		}
	}()

	osSig := make(chan os.Signal, 1)
	signal.Notify(osSig, os.Interrupt, syscall.SIGTERM, syscall.SIGQUIT)

	sig := <-osSig

	logger.Infof("Received signal %v\n", sig)

	server.Stop()
}

func getConnFromFD() (net.Conn, error) {
	file := os.NewFile(3, "")
	defer file.Close()
	if err := syscall.SetNonblock(int(file.Fd()), true); err != nil {
		return nil, err
	}
	conn, err := net.FileConn(file)
	if err != nil {
		return nil, err
	}
	tcpConn, ok := conn.(*net.TCPConn)
	if !ok {
		conn.Close()
		return nil, fmt.Errorf("Cannot get TCPConn from file")
	}
	return &util.TimeoutConn{FDConn: tcpConn, Timeout: constConnTimeout}, nil
}
