package backend

import (
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/ec2metadata"
	"github.com/aws/aws-sdk-go/aws/session"
	"github.com/aws/aws-sdk-go/service/lambda"
	"github.com/aws/aws-sdk-go/service/lambda/lambdaiface"
	"github.com/aws/aws-sdk-go/service/sqs"
	"github.com/aws/aws-sdk-go/service/ssm"
	"github.com/pkg/errors"
)

// Client contains methods for AWS interactions to allow easier data mocking for tests.
// If this interface changes, counterfeiter must be re-run using `make mocks`
type Client interface {
	SSMGetParameters(input *ssm.GetParametersInput) (*ssm.GetParametersOutput, error)
	EC2MGetInstanceIdentityDocument() (ec2metadata.EC2InstanceIdentityDocument, error)
	GetLambdaClient() lambdaiface.LambdaAPI
}

type client struct {
	sqs    *sqs.SQS
	ssm    *ssm.SSM
	ec2m   *ec2metadata.EC2Metadata
	lambda *lambda.Lambda
}

// New generates all the AWS clients required for this service
func New(region string) (Client, error) {
	sess, err := session.NewSession(&aws.Config{
		Region: aws.String(region),
	})

	if err != nil {
		return nil, errors.Wrap(err, "Error creating AWS session")
	}

	return &client{
		sqs:    sqs.New(sess),
		ssm:    ssm.New(sess),
		ec2m:   ec2metadata.New(sess),
		lambda: lambda.New(sess),
	}, nil
}

// GetLambdaClient returns the lambda client for use in twirp lambda transport
func (c *client) GetLambdaClient() lambdaiface.LambdaAPI {
	return c.lambda
}

// SSMGetParameters calls the underlying GetParameters from the ssm backend
func (c *client) SSMGetParameters(input *ssm.GetParametersInput) (*ssm.GetParametersOutput, error) {
	return c.ssm.GetParameters(input)
}

// EC2MGetInstanceIdentityDocument calls the underlying MGetInstanceIdentityDocument from the ec2metadata backend
func (c *client) EC2MGetInstanceIdentityDocument() (ec2metadata.EC2InstanceIdentityDocument, error) {
	return c.ec2m.GetInstanceIdentityDocument()
}
