package director

import (
	"net"
	"os"

	"code.justin.tv/amzn/TwirpGoLangAWSTransports/lambda"
	"code.justin.tv/event-engineering/golibs/pkg/logging"
	controlRPC "code.justin.tv/event-engineering/moonlight-api/pkg/rpc/control"
	"code.justin.tv/event-engineering/moonlight-rtmp/pkg/aws"
	awsBackend "code.justin.tv/event-engineering/moonlight-rtmp/pkg/aws/backend"
	"code.justin.tv/event-engineering/moonlight-rtmp/pkg/rtmp"
)

type Director interface {
	Start()
	Stop()
	Subscribe() chan rtmp.Event
}

type director struct {
	ID                 string
	logger             logging.Logger
	conn               net.Conn
	internalListenAddr string
	rtmpServer         rtmp.Server
	events             chan (rtmp.Event)
	stopped            bool
	moonlightClient    controlRPC.InternalController
}

func NewServer(conn net.Conn, internalListenAddr string, internalLambdaARN string, ab awsBackend.Client, logger logging.Logger) (Director, error) {
	var serverID string
	awsClient := aws.New(ab, logger)

	identity, err := awsClient.EC2MGetInstanceIdentityDocument()
	if err != nil {
		logger.Warnf("Failed to query instance metadata, this is usually because you're running outside of AWS, %v", err)
	}

	if identity.InstanceID == "" {
		serverID, err = os.Hostname()
		if err != nil {
			return nil, err
		}
	} else {
		serverID = identity.InstanceID
	}

	lambdaTransport := lambda.NewClient(awsClient.GetLambdaClient(), internalLambdaARN)
	moonlightClient := controlRPC.NewInternalControllerJSONClient("https://www.doesnt.matter.because.its.actually.lambda.transport.twitch.tv", lambdaTransport)

	sv := &director{
		ID:                 serverID,
		logger:             logger,
		conn:               conn,
		internalListenAddr: internalListenAddr,
		stopped:            false,
		moonlightClient:    moonlightClient,
	}

	return sv, nil
}

func (s *director) Start() {
	sv, err := rtmp.NewServer(s.conn, s.internalListenAddr, s.logger)
	if err != nil {
		s.logger.Errorf("Error handling RTMP %v", err)
		panic(err)
	}

	s.rtmpServer = sv
	s.events = s.rtmpServer.Subscribe()

	s.rtmpServer.Start()
	go s.handleEvents()
}

func (s *director) Stop() {
	s.stopped = true
	s.rtmpServer.Stop()
}

func (s *director) Subscribe() chan rtmp.Event {
	return s.rtmpServer.Subscribe()
}
