package router

// Route incoming RTMP play requests to the right internal worker that's serving that RTMP, if nothing is found then disconnect

import (
	"fmt"
	"net"

	"code.justin.tv/amzn/TwirpGoLangAWSTransports/lambda"
	gortmp "code.justin.tv/event-engineering/gortmp/pkg/rtmp"
	controlRPC "code.justin.tv/event-engineering/moonlight-api/pkg/rpc/control"
	"code.justin.tv/event-engineering/moonlight-rtmp/pkg/aws"
	awsBackend "code.justin.tv/event-engineering/moonlight-rtmp/pkg/aws/backend"
	rtmp "code.justin.tv/event-engineering/rtmp/pkg/common"
	player "code.justin.tv/event-engineering/rtmp/pkg/player"
	"github.com/sirupsen/logrus"
	goctx "golang.org/x/net/context"
)

func Create(conn net.Conn, internalLambdaARN string, ab awsBackend.Client, errCh chan error, logger logrus.FieldLogger) {
	awsClient := aws.New(ab, logger)
	lambdaTransport := lambda.NewClient(awsClient.GetLambdaClient(), internalLambdaARN)
	moonlightClient := controlRPC.NewInternalControllerJSONClient("https://www.doesnt.matter.because.its.actually.lambda.transport.twitch.tv", lambdaTransport)

	serverHandler := &rtmpServerHandler{
		logger:          logger,
		streams:         make(map[gortmp.MediaStream]map[string]bool),
		moonlightClient: moonlightClient,
		conn:            gortmp.NewBasicConn(conn),
	}
	ms := gortmp.NewMediaServer(serverHandler)
	serverHandler.mediaServer = ms

	var ctx = goctx.Background()

	logger.Infof("Accepted connection from (%s)", conn.RemoteAddr())

	go func() {
		if err := ms.ServeRTMP(ctx, conn); err != nil {
			logger.Warnf("Error service RTMP: %v", err)
		}

		conn.Close()
		errCh <- nil
	}()
}

type rtmpServerHandler struct {
	mediaServer     *gortmp.MediaServer
	moonlightClient controlRPC.InternalController
	streams         map[gortmp.MediaStream]map[string]bool
	logger          logrus.FieldLogger
	conn            gortmp.BasicConn
}

func (handler *rtmpServerHandler) Handle(ctx goctx.Context, r gortmp.Receiver, m gortmp.Message) error {
	if cmd, ok := m.(gortmp.Command); ok {
		return handler.HandleCommand(ctx, r, cmd)
	}
	return r.Handle(m)
}

func (handler *rtmpServerHandler) HandleCommand(ctx goctx.Context, r gortmp.Receiver, cmd gortmp.Command) error {
	switch cmd := cmd.(type) {
	case gortmp.PlayCommand:
		return handler.HandlePlay(ctx, r, cmd)
	}
	return r.Handle(cmd)
}

func (handler *rtmpServerHandler) Unpublish() {
	cmd := gortmp.OnStatusCommand{
		Info: gortmp.Status{Level: "status", Code: "NetStream.Unpublish.Success", Description: "Stop publishing"},
	}
	sendErr := rtmp.Send(handler.conn, cmd)
	if sendErr != nil {
		handler.logger.Warnf("Error sending message to client %v", sendErr)
	}

	handler.Close()
}

func (handler *rtmpServerHandler) PlayError(transactionID uint32) {
	cmd := gortmp.OnStatusCommand{
		Info: gortmp.NetStreamPlayInfo{
			Status: gortmp.Status{Level: "error", Code: "NetStream.Play.StreamNotFound", Description: "Stream Not Found"},
		},
	}
	sendErr := rtmp.Send(handler.conn, cmd)
	if sendErr != nil {
		handler.logger.Warnf("Error sending message to client %v", sendErr)
	}
}

func (handler *rtmpServerHandler) Close() {
	close := gortmp.RawCommand{
		Name: "close",
	}

	sendErr := rtmp.Send(handler.conn, close)
	if sendErr != nil {
		handler.logger.Warnf("Error sending message to client %v", sendErr)
	}
}

func (handler *rtmpServerHandler) HandlePlay(ctx goctx.Context, r gortmp.Receiver, play gortmp.PlayCommand) error {
	// Look up RTMP key from the db, play.Name is the RTMP key
	resp, err := handler.moonlightClient.GetRTMPSourceByRTMPKey(ctx, &controlRPC.GetRTMPSourceByRTMPKeyReq{
		RtmpKey: play.Name,
	})

	if err != nil {
		handler.PlayError(play.TransactionID)
		return fmt.Errorf("Could not find stream with key %v", play.Name)
	}

	listenAddr := resp.RtmpSource.GetInternalAddr()

	// There's no stream so die gracefully?
	if listenAddr == "" {
		handler.PlayError(play.TransactionID)
		return fmt.Errorf("Stream not found %v", play.Name)
	}

	ms := gortmp.NewMediaStream(ctx, play.Name)

	errCh := make(chan error)

	handler.logger.Warnf("Attempting to play back stream %v from %v", play.Name, listenAddr)
	pl := player.NewPlayer(errCh, handler.logger)

	err = pl.Play(fmt.Sprintf("rtmp://%v/app/%v", listenAddr, play.Name), ms)
	if err != nil {
		handler.PlayError(play.TransactionID)
		return fmt.Errorf("Stream unplayable %v - %v", play.Name, err)
	}

	err = handler.mediaServer.AddStream(ctx, ms)
	if err != nil {
		handler.PlayError(play.TransactionID)
		return fmt.Errorf("Could not add stream to mediaServer %v - %v", play.Name, err)
	}

	handler.streams[ms] = make(map[string]bool)
	handler.streams[ms][play.Name] = true

	go func() {
		err := <-errCh
		if err != nil {
			handler.logger.Warnf("Closing playback of stream %v due to playback error %v", play.Name, err)
			handler.Unpublish()
		}
	}()

	return r.Handle(play)
}

func (handler *rtmpServerHandler) OnMediaStreamCreated(ctx goctx.Context, stream gortmp.MediaStream) {
	// noop
}

func (handler *rtmpServerHandler) OnMediaStreamDestroyed(ctx goctx.Context, stream gortmp.MediaStream) {
	// noop
}
