package rtmp

import (
	"code.justin.tv/event-engineering/golibs/pkg/logging"
	rtmpctx "code.justin.tv/event-engineering/gortmp/pkg/context"
	gortmp "code.justin.tv/event-engineering/gortmp/pkg/rtmp"
	internal "code.justin.tv/event-engineering/moonlight-api/pkg/rpc"
	"code.justin.tv/event-engineering/rtmp/pkg/server"
	"context"
	"net"
)

type Event struct {
	StreamID string
	Type     internal.RTMPEvent
}

type Server interface {
	Start()
	Stop()
	Subscribe() chan Event
	GetInternaRTMPAddr() string
}

type rtmpServer struct {
	logger           logging.Logger
	ln               net.Listener
	subs             map[chan Event]bool
	conn             net.Conn
	internalListenIP string
	stopped          bool
}

func NewServer(conn net.Conn, internalListenIP string, logger logging.Logger) (Server, error) {
	sv := &rtmpServer{
		logger:           logger,
		subs:             make(map[chan Event]bool),
		conn:             conn,
		internalListenIP: internalListenIP,
		stopped:          false,
	}

	return sv, nil
}

func (s *rtmpServer) GetInternaRTMPAddr() string {
	return s.ln.Addr().String()
}

func (s *rtmpServer) Start() {
	go s.start()
}

func (s *rtmpServer) Stop() {
	s.stopped = true
	for ch := range s.subs {
		close(ch)
	}

	// TODO, do this more gracefully
	if s.ln != nil {
		s.ln.Close()
	}
}

func (s *rtmpServer) start() {
	ms := server.NewMediaServer(&handler{
		logger: s.logger,
		server: s,
	})

	sv := server.NewServer(server.ServerConfig{
		Handler: ms,
	})

	// Set up an internal only listener on an ephemeral port so that this stream can be found
	ln, err := net.Listen("tcp", net.JoinHostPort(s.internalListenIP, "0"))
	if err != nil {
		return
	}

	s.ln = ln

	go func(listener net.Listener) {
		if err := sv.Serve(listener); err != nil {
			s.logger.Warnf("Server error: %s", err)
		} else {
			s.logger.Infof("Server exited cleanly: %s", err)
		}
	}(ln)

	// Handle the incoming connection passed to us from the master process
	if err := sv.HandleConn(s.conn); err != nil {
		s.logger.Errorf("Server error: %s", err)
	} else {
		s.logger.Infof("Server exited cleanly: %s", err)
	}
}

func (s *rtmpServer) Subscribe() chan Event {
	ch := make(chan Event)
	s.subs[ch] = false
	return ch
}

func (s *rtmpServer) sendEvent(event Event) {
	for ch := range s.subs {
		go func(channel chan Event, ev Event) {
			if s.stopped {
				return
			}
			channel <- ev
		}(ch, event)
	}
}

type handler struct {
	logger logging.Logger
	server *rtmpServer
}

func (h *handler) OnMediaStreamCreated(ctx context.Context, ms gortmp.MediaStream) {
	h.logger.Info("MediaStream Created")
	streamName, ok := rtmpctx.GetStreamName(ctx)
	if !ok {
		h.logger.Error("Failed to get stream name")
	} else {
		h.server.sendEvent(Event{
			StreamID: streamName,
			Type:     internal.RTMPEvent_StreamStarted,
		})
		go h.server.trackStarvation(ctx, ms, streamName)
		go h.server.trackBitrate(ctx, ms, streamName)
	}
}

func (h *handler) OnMediaStreamDestroyed(ctx context.Context, ms gortmp.MediaStream) {
	h.logger.Info("MediaStream Destroyed")
	streamName, ok := rtmpctx.GetStreamName(ctx)
	if !ok {
		h.logger.Error("Failed to get stream name")
	} else {
		h.server.sendEvent(Event{
			StreamID: streamName,
			Type:     internal.RTMPEvent_StreamDisconnected,
		})
	}
}

func (h *handler) Handle(ctx context.Context, r gortmp.Receiver, msg gortmp.Message) error {
	return r.Handle(msg)
}
