package rtmp

import (
	gortmp "code.justin.tv/event-engineering/gortmp/pkg/rtmp"
	internal "code.justin.tv/event-engineering/moonlight-api/pkg/rpc"
	"container/list"
	"context"
	"time"
)

func (s *rtmpServer) trackStarvation(ctx context.Context, ms gortmp.MediaStream, streamName string) {
	s.logger.Info("start starvation tracking")
	defer s.logger.Info("end starvation tracking")

	ch, err := ms.Subscribe()
	if err != nil {
		s.logger.Errorf("stream subscribe failed")
		return
	}

	defer ms.Unsubscribe(ch)
	ticker := time.NewTicker(5 * time.Second)
	defer ticker.Stop()

	var curTs, lastTs int64
	var lastClock time.Time
	history := list.New()
	var starved = false

	for {
		if ms.IsClosed() {
			break
		}
		select {
		case tag, ok := <-ch:
			if ok {
				if tag.Type == gortmp.VIDEO_TYPE {
					curTs = int64(tag.Timestamp)
				}
			} else {
				return
			}
		case <-ticker.C:
			if lastClock.IsZero() {
				lastTs = curTs
				lastClock = time.Now()
			} else {
				streamDelta := (curTs - lastTs)
				clockDelta := time.Since(lastClock).Nanoseconds() / 1000000
				s.logger.Debugf("streamDelta=%d,clockDelta=%d", streamDelta, clockDelta)
				lastTs = curTs
				lastClock = time.Now()
				history.PushFront(clockDelta - streamDelta)

				if history.Len() > 25 {
					history.Remove(history.Back())
				}

				var sum int64

				for e := history.Front(); e != nil; e = e.Next() {
					sum += e.Value.(int64)
				}

				isStarved := float64(sum)/float64(clockDelta)/float64(history.Len()) > 0.05

				if isStarved && !starved {
					// Starvation started
					s.sendEvent(Event{
						StreamID: streamName,
						Type:     internal.RTMPEvent_StreamStarvationStart,
					})
					starved = true
					// this mimics the behavior of the wowza starvation tracker
					s.logger.Infof("sum = %d, len = %d", sum, history.Len())
					s.logger.Infof("starved = %t", isStarved)
					s.logger.Infof("starvation = %d", sum/int64(history.Len()))
				} else if !isStarved && starved {
					// Starvation ended
					s.sendEvent(Event{
						StreamID: streamName,
						Type:     internal.RTMPEvent_StreamStarvationEnd,
					})
					starved = false
					// this mimics the behavior of the wowza starvation tracker
					s.logger.Infof("sum = %d, len = %d", sum, history.Len())
					s.logger.Infof("starved = %t", isStarved)
					s.logger.Infof("starvation = %d", sum/int64(history.Len()))
				}
			}
		}
	}
}
