package obs

import (
	"encoding/json"
	"fmt"
	"sync"

	"code.justin.tv/event-engineering/golibs/pkg/logging"
	"github.com/google/uuid"
	"golang.org/x/net/websocket"
)

// A Client connects to a obs-studio websocket to get event and
// perform request on OBS instance remotely
type client struct {
	eventChannelsLock sync.RWMutex
	eventChannels     map[chan Event]interface{}

	wg sync.WaitGroup
	ws *websocket.Conn

	requestQueue    chan trackedRequest
	frames          chan []byte
	trackedRequests map[string]trackedRequest
	logger          logging.Logger
	stopped         bool
	started         bool
	ip              string
	port            int

	errCh chan error
}

// NewClient connects to an obs websocket instance.
func NewClient(ip string, port int, logger logging.Logger) Client {
	c := &client{
		requestQueue:    make(chan trackedRequest),
		trackedRequests: make(map[string]trackedRequest),
		logger:          logger,
		eventChannels:   make(map[chan Event]interface{}),
		ip:              ip,
		port:            port,
		errCh:           make(chan error),
	}

	return c
}

func (c *client) Go() (chan error, error) {
	c.stopped = false

	ws, err := websocket.Dial(fmt.Sprintf("ws://%v:%v/", c.ip, c.port), "", fmt.Sprintf("http://%v:%v/", c.ip, c.port))

	if err != nil {
		return nil, err
	}

	c.logger.Debug("Connected successfully")

	c.ws = ws

	go c.writeLoop()

	go c.readLoop()

	return c.errCh, nil
}

func (c *client) handleResponse(frame []byte) {
	// handle response
	var raw rawResponse
	err := json.Unmarshal(frame, &raw)
	if err != nil {
		c.logger.Warnf("Failed to unmarshal websocket message, %v", err)
		return
	}

	// This is in response to a request
	if raw.MessageID != "" {
		req, ok := c.trackedRequests[raw.MessageID]
		if ok == false {
			c.logger.Warnf("Received unknown message ID %v | %s", raw.MessageID, frame)
			return
		}

		// Grab the type that we expect the response to be for this message so we can unmarshall it
		response := req.getResponseType()
		err = json.Unmarshal(frame, &response)

		if err != nil {
			c.logger.Warnf("Error unmarshalling message: %v | %s", err, frame)
			return
		}

		respChan := req.getResponseChannel()
		respChan <- response
		delete(c.trackedRequests, raw.MessageID)
		close(respChan)
		return
	}

	// This is an event triggered by OBS
	if raw.UpdateType != "" {
		ev, err := unmarshalEvent(frame, raw.UpdateType)
		if err != nil {
			c.logger.Warn(err)
			return
		}

		c.eventChannelsLock.RLock()
		defer c.eventChannelsLock.RUnlock()
		for ch := range c.eventChannels {
			ch <- ev
		}

		return
	}

	c.logger.Warnf("Unhandled message %s", frame)
}

func (c *client) writeLoop() {
	for {
		if c.stopped {
			return
		}

		select {
		case f := <-c.frames:
			c.handleResponse(f)
		case r, ok := <-c.requestQueue:
			if !ok {
				if c.stopped {
					if c.ws != nil {
						c.ws.Close()
					}
					return
				}

				c.logger.Warn("Failed to read request from channel")
				continue
			}
			// Create a UID for this request
			messageID := uuid.New().String()
			r.setMessageID(messageID)

			// Store the request so we can reference it in the response we expect from the websocket
			c.trackedRequests[messageID] = r

			// Send it!
			err := websocket.JSON.Send(c.ws, r)

			if err != nil {
				c.logger.Warnf("Error sending websocket message %v", err)
				c.errCh <- err
				return
			}
		}
	}
}

func (c *client) readLoop() {
	c.wg.Add(1)
	// We need this in case we were never able to connect in the first place
	c.started = true
	defer c.wg.Done()

	c.frames = make(chan []byte)

	for {
		if c.stopped {
			return
		}

		// read bytes from the socket
		frame := make([]byte, 0, 1024)
		err := websocket.Message.Receive(c.ws, &frame)
		if err != nil {
			if c.stopped {
				return
			}

			c.logger.Warnf("Error receiving data on the socket %v", err)
			c.errCh <- err
			return
		}
		// send bytes to the channel to be handled by the main loop
		c.frames <- frame
	}
}

// Close closes all Event handlers and closes the websocket
func (c *client) Close() error {
	if c.stopped {
		// Close has already been called
		return nil
	}
	c.stopped = true

	close(c.errCh)
	close(c.requestQueue)
	if c.started {
		// wait to be done
		c.wg.Wait()
	}

	c.eventChannelsLock.Lock()
	defer c.eventChannelsLock.Unlock()
	for ch := range c.eventChannels {
		close(ch)
	}

	return nil
}

// Subscribe subscribes to the websocket to receive updates
func (c *client) Subscribe() chan Event {
	c.eventChannelsLock.RLock()
	defer c.eventChannelsLock.RUnlock()
	ch := make(chan Event)
	c.eventChannels[ch] = "lol"
	return ch
}

// Unsubscribe ussubscribes the supplied event channel
func (c *client) Unsubscribe(ch chan Event) {
	c.eventChannelsLock.RLock()
	defer c.eventChannelsLock.RUnlock()
	if _, ok := c.eventChannels[ch]; ok {
		close(ch)
		delete(c.eventChannels, ch)
	} else {
		c.logger.Warn("Attempted to unsubscribe event channel, but couldn't find it in the list")
	}
}
