package obs

import (
	"encoding/json"
	"fmt"
	"reflect"
	"time"
)

type rawEvent struct {
	StreamTC   string `json:"stream-timecode"`
	RecTC      string `json:"rec-timecode"`
	UpdateType string `json:"update-type"`
}

type Event interface {
	Type() string
	StreamTimecode() (time.Duration, bool)
	RecordTimecode() (time.Duration, bool)
}

func (e rawEvent) Type() string {
	return e.UpdateType
}

func (e rawEvent) StreamTimecode() (time.Duration, bool) {
	var duration time.Duration

	if len(e.StreamTC) > 0 {
		var err error
		duration, err = parseTC(e.StreamTC)
		if err != nil {
			return 0, false
		}
	}

	if duration < 0 {
		return 0, false
	}
	return duration, true
}

func (e rawEvent) RecordTimecode() (time.Duration, bool) {
	var duration time.Duration

	if len(e.RecTC) > 0 {
		var err error
		duration, err = parseTC(e.RecTC)
		if err != nil {
			return 0, false
		}
	}

	if duration < 0 {
		return 0, false
	}
	return duration, true
}

func parseTC(tc string) (time.Duration, error) {
	if len(tc) == 0 {
		return -1, fmt.Errorf("Invalid timecode")
	}
	var h, m, s, ms int
	_, err := fmt.Sscanf(tc, "%d:%d:%d.%d", &h, &m, &s, &ms)
	if err != nil {
		return -1, fmt.Errorf("Invalid Timecode '%s': %v", tc, err)
	}

	return time.Duration(h)*time.Hour + time.Duration(m)*time.Minute + time.Duration(s)*time.Second + time.Duration(ms)*time.Millisecond, nil
}

func unmarshalEvent(data []byte, eventType string) (Event, error) {
	// Grab the type from the list of supported types
	evType, ok := supportedEvents[eventType]
	if ok == false {
		return nil, fmt.Errorf("Unknown event type %v", eventType)
	}

	// Reflect a new instance of that type
	ev := reflect.New(evType).Interface()

	// Unmarshall data into the instance of the type we expect the message to be
	err := json.Unmarshal(data, &ev)
	if err != nil {
		return nil, err
	}

	return ev.(Event), nil
}

var supportedEvents map[string]reflect.Type

type EventSwitchScenes struct {
	SceneName string `json:"scene-name"`
	rawEvent
}

type EventScenesChanged struct {
	rawEvent
}

type EventSourceOrderChanged struct {
	SceneName string `json:"scene-name"`
	rawEvent
}

type EventSceneItemAdded struct {
	SceneName string `json:"scene-name"`
	ItemName  string `json:"item-name"`
	rawEvent
}

type EventSceneItemRemoved struct {
	SceneName string `json:"scene-name"`
	ItemName  string `json:"item-name"`
	rawEvent
}

type EventStreamStatus struct {
	Streaming        bool    `json:"streaming"`
	Recording        bool    `json:"recording"`
	PreviewOnly      bool    `json:"preview-only"`
	BytesPerSec      int     `json:"bytes-per-sec"`
	KBitsPerSec      int     `json:"kbits-per-sec"`
	Strain           float64 `json:"strain"`
	TotalStreamTime  int     `json:"total-stream-time"`
	NumTotalFrames   int     `json:"num-total-frames"`
	NumDroppedFrames int     `json:"num-dropped-frames"`
	Fps              float64 `json:"fps"`
	rawEvent
}

type OBSStats struct {
	FPS                 float64 `json:"fps"`
	RenderTotalFrames   int     `json:"render-total-frames"`
	RenderMissedFrames  int     `json:"render-missed-frames"`
	OutputTotalFrames   int     `json:"output-total-frames"`
	OutputSkippedFrames int     `json:"output-skipped-frames"`
	AverageFrameTime    float64 `json:"average-frame-time"`
	CPUUsage            float64 `json:"cpu-usage"`
	MemoryUsage         float64 `json:"memory-usage"`
	FreeDiskSpace       float64 `json:"free-disk-space"`
}

type EventHeartbeat struct {
	Pulse             bool     `json:"pulse"`
	CurrentProfile    string   `json:"current-profile"`
	CurrentScene      string   `json:"current-scene"`
	Streaming         bool     `json:"streaming"`
	TotalStreamTime   int      `json:"total-stream-time"`
	TotalStreamBytes  int      `json:"total-stream-bytes"`
	TotalStreamFrames int      `json:"total-stream-frames"`
	Recording         bool     `json:"recording"`
	TotalRecordTime   int      `json:"total-record-time"`
	TotalRecordBytes  int      `json:"total-record-bytes"`
	TotalRecordFrames int      `json:"total-record-frames"`
	Stats             OBSStats `json:"stats"`
	rawEvent
}

type EventSourceMuteStateChanged struct {
	SourceName string `json:"sourceName"`
	Muted      bool   `json:"muted"`
	rawEvent
}

func init() {
	supportedEvents = map[string]reflect.Type{
		"SwitchScenes":           reflect.TypeOf(EventSwitchScenes{}),
		"ScenesChanged":          reflect.TypeOf(EventScenesChanged{}),
		"SourceOrderChanged":     reflect.TypeOf(EventSourceOrderChanged{}),
		"SceneItemAdded":         reflect.TypeOf(EventSceneItemAdded{}),
		"SceneItemRemoved":       reflect.TypeOf(EventSceneItemRemoved{}),
		"StreamStatus":           reflect.TypeOf(EventStreamStatus{}),
		"Heartbeat":              reflect.TypeOf(EventHeartbeat{}),
		"SourceMuteStateChanged": reflect.TypeOf(EventSourceMuteStateChanged{}),
	}
}
