package obs

import (
	"fmt"
)

type Client interface {
	Subscribe() chan Event
	Unsubscribe(ch chan Event)
	Close() error
	Go() (chan error, error)
	SetCurrentScene(sceneIdentifier string) error
	GetSceneList() (*GetSceneListResponse, error)
	StartStream() error
	StopStream() error
	SetStreamKey(streamKey string) error
	GetStreamStatus() (*GetStreamingStatusResponse, error)
	SetHeartbeat(enable bool) error
	SetMute(source string, mute bool) error
	GetMute(source string) (*GetMuteResponse, error)
	SetBrowserSourceProperties(properties *SetBrowserSourcePropertiesRequest) error
	SetSceneItemProperties(properties *SetSceneItemPropertiesRequest) error
}

func (c *client) GetSceneList() (*GetSceneListResponse, error) {
	resp, err := c.submitRequest(&requestBase{
		RequestType:  "GetSceneList",
		responseType: &GetSceneListResponse{},
	})

	if err != nil {
		return nil, err
	}
	respCorrect, ok := resp.(*GetSceneListResponse)
	if !ok {
		return nil, fmt.Errorf("obsws: unexpected response from server: %#v", resp)
	}
	return respCorrect, nil
}

type setStreamSettings struct {
	requestBase
	Type     string         `json:"type"`
	Settings streamSettings `json:"settings"`
}

type streamSettings struct {
	StreamKey string `json:"key"`
}

type setCurrentSceneRequest struct {
	requestBase
	SceneName string `json:"scene-name"`
}

type setHeartbeatRequest struct {
	requestBase
	Enable bool `json:"enable"`
}

type setMuteRequest struct {
	requestBase
	Source string `json:"source"`
	Mute   bool   `json:"mute"`
}

type getMuteRequest struct {
	requestBase
	Source string `json:"source"`
}

func (c *client) SetCurrentScene(name string) error {
	_, err := c.submitRequest(&setCurrentSceneRequest{
		requestBase: requestBase{
			RequestType:  "SetCurrentScene",
			responseType: &rawResponse{},
		},
		SceneName: name,
	})
	return err
}

func (c *client) SetStreamKey(streamKey string) error {
	_, err := c.submitRequest(&setStreamSettings{
		requestBase: requestBase{
			RequestType:  "SetStreamSettings",
			responseType: &rawResponse{},
		},
		Type: "rtmp_common",
		Settings: streamSettings{
			StreamKey: streamKey,
		},
	})
	return err
}

func (c *client) GetStreamStatus() (*GetStreamingStatusResponse, error) {
	resp, err := c.submitRequest(&requestBase{
		RequestType:  "GetSceneList",
		responseType: &GetStreamingStatusResponse{},
	})

	if err != nil {
		return nil, err
	}
	statusResp, ok := resp.(*GetStreamingStatusResponse)
	if !ok {
		return nil, fmt.Errorf("obsws: unexpected response from server: %#v", resp)
	}

	return statusResp, nil
}

func (c *client) StartStream() error {
	_, err := c.submitRequest(&requestBase{
		RequestType:  "StartStreaming",
		responseType: &rawResponse{},
	})
	return err
}

func (c *client) StopStream() error {
	_, err := c.submitRequest(&requestBase{
		RequestType:  "StopStreaming",
		responseType: &rawResponse{},
	})
	return err
}

func (c *client) SetHeartbeat(enable bool) error {
	_, err := c.submitRequest(&setHeartbeatRequest{
		requestBase: requestBase{
			RequestType:  "SetHeartbeat",
			responseType: &rawResponse{},
		},
		Enable: enable,
	})
	return err
}

func (c *client) SetMute(source string, mute bool) error {
	_, err := c.submitRequest(&setMuteRequest{
		requestBase: requestBase{
			RequestType:  "SetMute",
			responseType: &rawResponse{},
		},
		Source: source,
		Mute:   mute,
	})
	return err
}

func (c *client) GetMute(source string) (*GetMuteResponse, error) {
	resp, err := c.submitRequest(&getMuteRequest{
		requestBase: requestBase{
			RequestType:  "GetMute",
			responseType: &GetMuteResponse{},
		},
		Source: source,
	})

	getMuteResp, ok := resp.(*GetMuteResponse)
	if !ok {
		return nil, fmt.Errorf("obsws - getmute: unexpected response from server: %#v", resp)
	}

	return getMuteResp, err
}

func (c *client) SetBrowserSourceProperties(properties *SetBrowserSourcePropertiesRequest) error {
	properties.requestBase = requestBase{
		RequestType:  "SetBrowserSourceProperties",
		responseType: &rawResponse{},
	}
	_, err := c.submitRequest(properties)
	return err
}

func (c *client) SetSceneItemProperties(properties *SetSceneItemPropertiesRequest) error {
	properties.requestBase = requestBase{
		RequestType:  "SetSceneItemProperties",
		responseType: &rawResponse{},
	}
	_, err := c.submitRequest(properties)
	return err
}
