package obs

import (
	"errors"
)

type SceneItemBoundsType string

const (
	BoundsStretch       SceneItemBoundsType = "OBS_BOUNDS_STRETCH"
	BoundsScaleInner    SceneItemBoundsType = "OBS_BOUNDS_SCALE_INNER"
	BoundsScaleOuter    SceneItemBoundsType = "OBS_BOUNDS_SCALE_OUTER"
	BoundsScaleToWidth  SceneItemBoundsType = "OBS_BOUNDS_SCALE_TO_WIDTH"
	BoundsScaleToHeight SceneItemBoundsType = "OBS_BOUNDS_SCALE_TO_HEIGHT"
	BoundsMaxOnly       SceneItemBoundsType = "OBS_BOUNDS_MAX_ONLY"
	BoundsNone          SceneItemBoundsType = "OBS_BOUNDS_NONE"
)

// https://github.com/obsproject/obs-studio/blob/c938ea712bce0e9d8e0cf348fd8f77725122b9a5/libobs/obs-defs.h#L23
const (
	AlignmentCenter = 0
	AlignmentLeft   = 1 << 0
	AlignmentRight  = 1 << 1
	AlignmentTop    = 1 << 2
	AlignmentBottom = 1 << 3
)

type SceneItemPosition struct {
	X         int `json:"x"`
	Y         int `json:"y"`
	Alignment int `json:"alignment"`
}

type SceneItemScale struct {
	X int `json:"x"`
	Y int `json:"y"`
}

type SceneItemCrop struct {
	Top    int `json:"top"`
	Bottom int `json:"bottom"`
	Left   int `json:"left"`
	Right  int `json:"right"`
}

type SceneItemBounds struct {
	Type      SceneItemBoundsType `json:"type"`
	X         float64             `json:"x"`
	Y         float64             `json:"y"`
	Alignment int                 `json:"alignment"`
}

type SetSceneItemPropertiesRequest struct {
	requestBase
	SceneName string            `json:"scene-name"`
	Item      string            `json:"item"`
	Position  SceneItemPosition `json:"position"`
	Rotation  float64           `json:"rotation"`
	Scale     SceneItemScale    `json:"scale"`
	Crop      SceneItemCrop     `json:"crop"`
	Visible   bool              `json:"visible"`
	Locked    bool              `json:"locked"`
	Bounds    SceneItemBounds   `json:"bounds"`
}

type SetBrowserSourcePropertiesRequest struct {
	requestBase
	Source      string `json:"source"`
	IsLocalFile bool   `json:"is_local_file"`
	LocalFile   string `json:"local_file"`
	URL         string `json:"url"`
	CSS         string `json:"css"`
	Width       int    `json:"width"`
	Height      int    `json:"height"`
	FPS         int    `json:"fps"`
	Shutdown    bool   `json:"shutdown"`
	Render      bool   `json:"render"`
}

type requestBase struct {
	MessageID    string `json:"message-id"`
	RequestType  string `json:"request-type"`
	response     chan response
	responseType response
}

func (r *requestBase) setMessageID(ID string) {
	r.MessageID = ID
}

func (r *requestBase) getResponseChannel() chan response {
	return r.response
}

func (r *requestBase) setResponseChannel(rchan chan response) {
	r.response = rchan
}

func (r *requestBase) getResponseType() response {
	return r.responseType
}

func createRequest(name string, resp response) trackedRequest {
	return &requestBase{
		RequestType:  name,
		responseType: resp,
	}
}

type trackedRequest interface {
	setMessageID(uid string)
	getResponseChannel() chan response
	setResponseChannel(rchan chan response)
	getResponseType() response
}

func (c *client) submitRequest(r trackedRequest) (response, error) {
	rchan := make(chan response)
	r.setResponseChannel(rchan)
	c.requestQueue <- r
	resp, ok := <-rchan
	if !ok {
		return nil, errors.New("Expected response but didn't get one")
	}

	if err := resp.error(); err != nil {
		return nil, err
	}
	return resp, nil
}
