package auth

import (
	"context"
	"encoding/json"
	"log"
	"net/http"

	rpc "code.justin.tv/event-engineering/parsnip/pkg/rpc"
	svc "code.justin.tv/event-engineering/parsnip/pkg/svc"
	jwt "github.com/dgrijalva/jwt-go"
	"github.com/sirupsen/logrus"
)

type key int

const (
	// KeyAmznLogin is the context key for the amazon sso login
	keyAmznLogin key = iota
)

// WithDummyAuth wraps a http handler and adds dummy user info to the context
func WithDummyAuth(base http.Handler, userLogin string) http.Handler {
	return http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		// This is for use in the app
		ctx := context.WithValue(r.Context(), keyAmznLogin, userLogin)
		// This needs to be here to pass to the carrot control context for auditing
		ctx = svc.WithUserID(ctx, userLogin)

		r = r.WithContext(ctx)
		base.ServeHTTP(w, r)
	})
}

type ForbiddenError struct {
	Error string `json:"error"`
}

// WithAmznSSO wraps a http handler and adds amzn_sso_token info to the context
func WithAmznSSO(base http.Handler) http.Handler {
	return http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		ctx := r.Context()

		cookie, err := r.Cookie("amzn_sso_token")
		if err != nil {
			log.Printf("Error getting cookie amzn_sso_token %v\n", err)
			w.WriteHeader(401)
			return
		}

		// TrivEA is the only thing calling this function and token validation is done there, so we're just gonna parse it to get the username
		_, err = jwt.Parse(cookie.Value, func(token *jwt.Token) (interface{}, error) {
			if claims, ok := token.Claims.(jwt.MapClaims); ok {
				userID := claims["sub"].(string)
				// This is for use in the app
				ctx = context.WithValue(ctx, keyAmznLogin, userID)
				// This needs to be here to pass to the parsnip context for permissions
				ctx = svc.WithUserID(ctx, userID)
			}

			r = r.WithContext(ctx)

			return "", nil
		})

		if err != nil {
			log.Printf("Error parsing token %v\n", err)
		}

		base.ServeHTTP(w, r)
	})
}

// GetUser returns the user from the supplied context
func GetUser(ctx context.Context) string {
	return ctx.Value(keyAmznLogin).(string)
}

// User is just a simple struct representing the logged in user
type User struct {
	Login       string                       `json:"login"`
	Permissions *rpc.GetModuleAccessResponse `json:"permissions"`
}

// GetUserInfo is a tiny http handler that returns the login of the current users
func GetUserInfo(parsnipAPI rpc.Parsnip, logger logrus.FieldLogger) http.Handler {
	return http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		login := GetUser(r.Context())

		perms, err := parsnipAPI.GetModuleAccess(r.Context(), &rpc.GetModuleAccessRequest{})

		if err != nil {
			logger.WithError(err).Warn("Failed to get User Module Access")
		}

		w.WriteHeader(200)
		encoder := json.NewEncoder(w)
		err = encoder.Encode(User{Login: login, Permissions: perms})

		if err != nil {
			logger.WithError(err).Warn("Failed to encode user object")
		}
	})
}
