package app

import (
	"net/http"
	"strconv"

	"code.justin.tv/event-engineering/parsnip/app/api/auth"
	"code.justin.tv/event-engineering/parsnip/app/modules"
	parsnip "code.justin.tv/event-engineering/parsnip/pkg/rpc"
	"github.com/gobuffalo/packr"
	"github.com/gorilla/csrf"
	"github.com/sirupsen/logrus"
	goji "goji.io"
	"goji.io/pat"
)

type responseWriter struct {
	header        http.Header
	body          []byte
	headerWritten bool
	statusCode    int
}

func (w *responseWriter) Header() http.Header {
	if w.header == nil {
		w.header = http.Header{}
	}
	return w.header
}

func (w *responseWriter) Write(bytes []byte) (int, error) {
	if !w.headerWritten {
		w.WriteHeader(http.StatusOK)
	}
	if w.body == nil {
		w.body = make([]byte, 0)
	}
	w.body = append(w.body, bytes...)
	return len(bytes), nil
}

func (w *responseWriter) WriteHeader(statusCode int) {
	if w.headerWritten {
		return
	}
	w.statusCode = statusCode
	w.Header().Set("Status", strconv.Itoa(statusCode))
	w.headerWritten = true
}

type indexResolver struct {
	staticHandler http.Handler
	logger        logrus.FieldLogger
}

func (r *indexResolver) ServeHTTP(writer http.ResponseWriter, req *http.Request) {
	rw := &responseWriter{}
	r.staticHandler.ServeHTTP(rw, req)

	if rw.statusCode == http.StatusNotFound {
		req.URL.Path = "/"
		r.staticHandler.ServeHTTP(writer, req)
	} else {
		wh := writer.Header()
		rh := rw.Header()
		for k := range rh {
			wh.Set(k, rh.Get(k))
		}
		writer.WriteHeader(rw.statusCode)
		_, err := writer.Write(rw.body)
		if err != nil {
			r.logger.Error(err)
		}
	}
}

type csrfTokenInserter struct {
	baseHandler http.Handler
}

func (a csrfTokenInserter) ServeHTTP(w http.ResponseWriter, r *http.Request) {
	w.Header().Set("X-CSRF-Token", csrf.Token(r))
	a.baseHandler.ServeHTTP(w, r)
}

func New(logger logrus.FieldLogger, parsnipAPI parsnip.Parsnip, mainMux *goji.Mux, csrfAuthKey string) {
	handler := &Handler{
		Mux: mainMux,
	}

	if csrfAuthKey != "" {
		CSRF := csrf.Protect([]byte(csrfAuthKey), csrf.SameSite(csrf.SameSiteStrictMode), csrf.Path("/"))
		handler.Mux.Use(CSRF)
		handler.Mux.Use(func(h http.Handler) http.Handler {
			return &csrfTokenInserter{
				baseHandler: h,
			}
		})
	}

	// All API stuff goes under the /api path
	apiMux := goji.SubMux()
	handler.Mux.Handle(pat.New("/api/*"), http.StripPrefix("/api", apiMux))

	// Add all the module handlers
	modules.AddModules(apiMux, parsnipAPI, logger)

	// This is just a standalone endpoint to retrieve username info from the SSO token
	apiMux.Handle(pat.Get("/auth/info"), auth.GetUserInfo(parsnipAPI, logger))

	// Static web content is default
	staticHandler := http.FileServer(packr.NewBox("../ui/build"))

	// Make all requests that don't match something return index.html
	indexResolver := &indexResolver{
		logger:        logger,
		staticHandler: staticHandler,
	}

	handler.Mux.Handle(pat.Get("/*"), indexResolver)

	return
}
