package admin

import (
	"encoding/json"
	"io/ioutil"
	"net/http"

	"code.justin.tv/event-engineering/parsnip/app/util"
	parsnip "code.justin.tv/event-engineering/parsnip/pkg/rpc"
	"github.com/sirupsen/logrus"
	goji "goji.io"
	"goji.io/pat"
)

type handler struct {
	mux     *goji.Mux
	parsnip parsnip.Parsnip
	logger  logrus.FieldLogger
}

// NewAdminHandler will forward admin API requests to the parsnip service
func NewAdminHandler(parsnipAPI parsnip.Parsnip, logger logrus.FieldLogger) http.Handler {
	handler := &handler{
		mux:     goji.NewMux(),
		logger:  logger,
		parsnip: parsnipAPI,
	}

	// Get Auth token
	handler.mux.HandleFunc(pat.Get("/channel"), handler.ListChannels)
	handler.mux.HandleFunc(pat.Get("/channel/:channel_id_hash"), handler.GetChannel)
	handler.mux.HandleFunc(pat.Post("/channel"), handler.CreateChannel)
	handler.mux.HandleFunc(pat.Put("/channel/:channel_id_hash"), handler.UpdateChannel)
	handler.mux.HandleFunc(pat.Delete("/channel/:channel_id_hash"), handler.DeleteChannel)

	return handler.mux
}

func (s *handler) ListChannels(writer http.ResponseWriter, request *http.Request) {
	s.logger.Debug("Calling ListChannels")
	resp, err := s.parsnip.ListChannels(request.Context(), &parsnip.ListChannelsRequest{})

	if err != nil {
		s.logger.WithError(err).Warn("Failed to call ListChannels")
	}

	util.HandleResponse(writer, resp, err)
}

func (s *handler) GetChannel(writer http.ResponseWriter, request *http.Request) {
	channelIDHash := pat.Param(request, "channel_id_hash")

	s.logger.Debugf("Calling GetChannel with ID %v", channelIDHash)
	resp, err := s.parsnip.GetChannel(request.Context(), &parsnip.GetChannelRequest{
		ChannelIdHash: channelIDHash,
	})

	if err != nil {
		s.logger.WithError(err).Warn("Failed to call GetChannel")
	}

	util.HandleResponse(writer, resp, err)
}

func (s *handler) DeleteChannel(writer http.ResponseWriter, request *http.Request) {
	channelIDHash := pat.Param(request, "channel_id_hash")

	s.logger.Debugf("Calling DeleteChannel with ID %v", channelIDHash)
	resp, err := s.parsnip.DeleteChannel(request.Context(), &parsnip.DeleteChannelRequest{
		ChannelIdHash: channelIDHash,
	})

	if err != nil {
		s.logger.WithError(err).Warn("Failed to call DeleteChannel")
	}

	util.HandleResponse(writer, resp, err)
}

type createChannelRequest struct {
	ChannelID          string `json:"channel_id"`
	ChannelTitle       string `json:"channel_title"`
	ChannelDescription string `json:"channel_description"`
}

func (s *handler) CreateChannel(writer http.ResponseWriter, request *http.Request) {
	var req createChannelRequest
	bytes, err := ioutil.ReadAll(request.Body)
	if err != nil {
		util.HandleError(writer, http.StatusBadRequest, err)
		return
	}

	err = json.Unmarshal(bytes, &req)
	if err != nil {
		util.HandleError(writer, http.StatusBadRequest, err)
		return
	}

	s.logger.Debug("Calling CreateChannel")
	resp, err := s.parsnip.CreateChannel(request.Context(), &parsnip.CreateChannelRequest{
		ChannelId:          req.ChannelID,
		ChannelTitle:       req.ChannelTitle,
		ChannelDescription: req.ChannelDescription,
	})

	if err != nil {
		s.logger.WithError(err).Warn("Failed to call CreateChannel")
	}

	util.HandleResponse(writer, resp, err)
}

type updateChannelRequest struct {
	BindleLockID       string `json:"bindle_lock_id"`
	ChannelTitle       string `json:"channel_title"`
	ChannelDescription string `json:"channel_description"`
}

func (s *handler) UpdateChannel(writer http.ResponseWriter, request *http.Request) {
	channelIDHash := pat.Param(request, "channel_id_hash")

	var req updateChannelRequest
	bytes, err := ioutil.ReadAll(request.Body)
	if err != nil {
		util.HandleError(writer, http.StatusBadRequest, err)
		return
	}

	err = json.Unmarshal(bytes, &req)
	if err != nil {
		util.HandleError(writer, http.StatusBadRequest, err)
		return
	}

	s.logger.Debug("Calling UpdateChannel")
	resp, err := s.parsnip.UpdateChannel(request.Context(), &parsnip.UpdateChannelRequest{
		ChannelIdHash:      channelIDHash,
		BindleLockId:       req.BindleLockID,
		ChannelTitle:       req.ChannelTitle,
		ChannelDescription: req.ChannelDescription,
	})

	if err != nil {
		s.logger.WithError(err).Warn("Failed to call UpdateChannel")
	}

	util.HandleResponse(writer, resp, err)
}
