package question

import (
	"crypto/sha256"
	"encoding/json"
	"fmt"
	"io/ioutil"
	"net/http"

	auth "code.justin.tv/event-engineering/parsnip/app/api/auth"
	"code.justin.tv/event-engineering/parsnip/app/util"
	parsnip "code.justin.tv/event-engineering/parsnip/pkg/rpc"
	"github.com/sirupsen/logrus"
	goji "goji.io"
	"goji.io/pat"
)

type handler struct {
	mux     *goji.Mux
	parsnip parsnip.Parsnip
	logger  logrus.FieldLogger
}

func getLoginHash(login string) string {
	return fmt.Sprintf("%x", sha256.Sum256([]byte(login)))
}

// NewQuestionHandler will forward questions API requests to the parsnip service
func NewQuestionHandler(parsnipAPI parsnip.Parsnip, logger logrus.FieldLogger) http.Handler {
	handler := &handler{
		mux:     goji.NewMux(),
		logger:  logger,
		parsnip: parsnipAPI,
	}

	// Get Auth token
	handler.mux.HandleFunc(pat.Get("/:channel_id"), handler.GetQuestions)
	handler.mux.HandleFunc(pat.Post("/:channel_id"), handler.CreateQuestion)
	handler.mux.HandleFunc(pat.Get("/:channel_id/:question_id"), handler.GetQuestion)
	handler.mux.HandleFunc(pat.Delete("/:channel_id/:question_id"), handler.DeleteQuestion)
	handler.mux.HandleFunc(pat.Delete("/:channel_id"), handler.DeleteAllQuestions)
	handler.mux.HandleFunc(pat.Post("/:channel_id/:question_id/upvote"), handler.UpvoteQuestion)
	handler.mux.HandleFunc(pat.Post("/:channel_id/:question_id/answer"), handler.AnswerQuestion)

	return handler.mux
}

func (s *handler) GetQuestion(writer http.ResponseWriter, request *http.Request) {
	login := auth.GetUser(request.Context())
	loginHash := getLoginHash(login)
	channelID := pat.Param(request, "channel_id")
	questionID := pat.Param(request, "question_id")

	// Login is determined by the API since we're passing the request context through
	s.logger.Debug("Calling GetQuestions with user %v", login)
	resp, err := s.parsnip.GetQuestion(request.Context(), &parsnip.GetQuestionRequest{
		QuestionId: questionID,
		ChannelId:  channelID,
		HashUserId: loginHash,
	})

	if err != nil {
		s.logger.WithError(err).Warnf("Failed to get questions for channel: %s", channelID)
	}

	util.HandleResponse(writer, resp, err)
}

func (s *handler) GetQuestions(writer http.ResponseWriter, request *http.Request) {
	login := auth.GetUser(request.Context())
	channelID := pat.Param(request, "channel_id")
	loginHash := getLoginHash(login)

	// Login is determined by the API since we're passing the request context through
	s.logger.Debug("Calling GetQuestions with user %v", login)
	resp, err := s.parsnip.GetQuestions(request.Context(), &parsnip.GetQuestionsRequest{
		ChannelId:  channelID,
		HashUserId: loginHash,
	})

	if err != nil {
		s.logger.WithError(err).Warnf("Failed to get questions for channel: %s", channelID)
	}

	util.HandleResponse(writer, resp, err)
}

type createQuestionRequest struct {
	Description string `json:"description"`
}

func (s *handler) CreateQuestion(writer http.ResponseWriter, request *http.Request) {
	channelID := pat.Param(request, "channel_id")
	login := auth.GetUser(request.Context())
	loginHash := getLoginHash(login)

	var req createQuestionRequest
	bytes, err := ioutil.ReadAll(request.Body)
	if err != nil {
		util.HandleError(writer, http.StatusBadRequest, err)
		return
	}

	err = json.Unmarshal(bytes, &req)
	if err != nil {
		util.HandleError(writer, http.StatusBadRequest, err)
		return
	}

	s.logger.Debug("Calling CreateQuestion")
	resp, err := s.parsnip.CreateQuestion(request.Context(), &parsnip.CreateQuestionRequest{
		ChannelId:     channelID,
		HashCreatedBy: loginHash,
		Description:   req.Description,
	})

	if err != nil {
		s.logger.WithError(err).Warn("Failed to call CreateChannel")
	}

	util.HandleResponse(writer, resp, err)
}

func (s *handler) UpvoteQuestion(writer http.ResponseWriter, request *http.Request) {
	login := auth.GetUser(request.Context())
	loginHash := getLoginHash(login)
	channelID := pat.Param(request, "channel_id")
	questionID := pat.Param(request, "question_id")

	s.logger.Debug("Calling UpvoteQuestion")
	resp, err := s.parsnip.UpvoteQuestion(request.Context(), &parsnip.UpvoteQuestionRequest{
		ChannelId:  channelID,
		QuestionId: questionID,
		HashUserId: loginHash,
	})

	if err != nil {
		s.logger.WithError(err).Warn("Failed to call UpvoteQuestion")
	}

	util.HandleResponse(writer, resp, err)
}

func (s *handler) AnswerQuestion(writer http.ResponseWriter, request *http.Request) {
	channelID := pat.Param(request, "channel_id")
	questionID := pat.Param(request, "question_id")

	s.logger.Debug("Calling AnswerQuestion")
	resp, err := s.parsnip.AnswerQuestion(request.Context(), &parsnip.AnswerQuestionRequest{
		ChannelId:  channelID,
		QuestionId: questionID,
	})

	if err != nil {
		s.logger.WithError(err).Warn("Failed to call AnswerQuestion")
	}

	util.HandleResponse(writer, resp, err)
}

func (s *handler) DeleteQuestion(writer http.ResponseWriter, request *http.Request) {
	login := auth.GetUser(request.Context())
	loginHash := getLoginHash(login)
	channelID := pat.Param(request, "channel_id")
	questionID := pat.Param(request, "question_id")

	// Login is determined by the API since we're passing the request context through
	s.logger.Debug("Calling GetQuestions with user %v", login)
	resp, err := s.parsnip.DeleteQuestion(request.Context(), &parsnip.DeleteQuestionRequest{
		QuestionId: questionID,
		ChannelId:  channelID,
		HashUserId: loginHash,
	})

	if err != nil {
		s.logger.WithError(err).Warnf("Failed to delete question for channel: %s", channelID)
	}

	util.HandleResponse(writer, resp, err)
}

func (s *handler) DeleteAllQuestions(writer http.ResponseWriter, request *http.Request) {
	login := auth.GetUser(request.Context())
	loginHash := getLoginHash(login)
	channelID := pat.Param(request, "channel_id")

	// Login is determined by the API since we're passing the request context through
	s.logger.Debug("Calling GetQuestions with user %v", login)
	resp, err := s.parsnip.DeleteAllQuestions(request.Context(), &parsnip.DeleteAllQuestionsRequest{
		ChannelId:  channelID,
		HashUserId: loginHash,
	})

	if err != nil {
		s.logger.WithError(err).Warnf("Failed to delete all questions for channel: %s", channelID)
	}

	util.HandleResponse(writer, resp, err)
}
