import cdk = require('@aws-cdk/core');
import ec2 = require('@aws-cdk/aws-ec2');
import iam = require('@aws-cdk/aws-iam');
import ddb = require('@aws-cdk/aws-dynamodb');
import lambda = require('@aws-cdk/aws-lambda');

export interface ParsnipAPIProps {
  playbackAuthKeyParameterName: string
  adminBindleLockID: string
  tokenExpirySeconds: number
  acao: string
  publisherBindleLockID: string
  invokeFunctionRoles: string[]
}

export class ParsnipAPI extends cdk.Construct {
  ChannelsTable: ddb.Table
  QuestionsTable: ddb.Table
  ServiceLambda: lambda.Function

  constructor(scope: cdk.Construct, id: string, vpc: ec2.IVpc, props: ParsnipAPIProps) {
    super(scope, id);

    // Dynamo db table for tracking RTMP capture endpoints
    this.ChannelsTable = new ddb.Table(this, 'ChannelsTable', {
      tableName: 'parsnip-channels',
      partitionKey: { name: 'channel_id_hash', type: ddb.AttributeType.STRING },
      billingMode: ddb.BillingMode.PAY_PER_REQUEST,
    })

    this.ChannelsTable.addGlobalSecondaryIndex({
      indexName: 'idx_by_channel_id',
      partitionKey: { name: 'channel_id', type: ddb.AttributeType.STRING },
      projectionType: ddb.ProjectionType.ALL,
    })

    // Dynamo db table for Q&A questions for a channel
    this.QuestionsTable = new ddb.Table(this, 'QuestionsTable', {
      tableName: 'parsnip-questions',
      partitionKey: { name: 'question_id', type: ddb.AttributeType.STRING },
      billingMode: ddb.BillingMode.PAY_PER_REQUEST,
    })

    this.QuestionsTable.addGlobalSecondaryIndex({
      indexName: 'idx_by_channel_id',
      partitionKey: { name: 'channel_id', type: ddb.AttributeType.STRING },
      projectionType: ddb.ProjectionType.ALL,
    })

    // Lambda for the service API
    this.ServiceLambda = new lambda.Function(this, 'Lambda', {
      vpc: vpc,
      runtime: lambda.Runtime.GO_1_X,
      memorySize: 1024,
      timeout: cdk.Duration.seconds(5),
      handler: 'bin/parsnip-api/handler',
      code: lambda.Code.asset('../bin/parsnip-api.zip'),
      environment: {
        "channelsTableName": this.ChannelsTable.tableName,
        "questionsTableName": this.QuestionsTable.tableName,
        "playbackAuthKeyParameterName": props.playbackAuthKeyParameterName,
        "adminBindleLockID": props.adminBindleLockID,
        "publisherBindleLockID": props.publisherBindleLockID,
        "tokenExpirySeconds": props.tokenExpirySeconds.toString(),
        "acao": props.acao,
      }
    })

    // Allow things to call the function
    for (let roleArn of props.invokeFunctionRoles) {
      this.ServiceLambda.grantInvoke(new iam.ArnPrincipal(roleArn))
    }

    this.ServiceLambda.addToRolePolicy(new iam.PolicyStatement({
      effect: iam.Effect.ALLOW,
      actions: [
        "ssm:GetParameter",
        "ssm:GetParameters",
      ],
      resources: [
        cdk.Arn.format({
          partition: "aws",
          service: "ssm",
          account: cdk.Stack.of(this).account,
          region: cdk.Stack.of(this).region,
          resource: 'parameter',
          resourceName: props.playbackAuthKeyParameterName, // This created OOB and added manually to SSM
        }, cdk.Stack.of(this))
      ]
    }));

    this.ServiceLambda.addToRolePolicy(new iam.PolicyStatement({
      sid: 'DDBAccess',
      effect: iam.Effect.ALLOW,
      actions: [
        "dynamodb:PutItem",
        "dynamodb:DeleteItem",
        "dynamodb:UpdateItem",
        "dynamodb:GetItem",
        "dynamodb:Scan",
        "dynamodb:Query"
      ],
      resources: [
        this.ChannelsTable.tableArn,
        this.ChannelsTable.tableArn + '/*',
        this.QuestionsTable.tableArn,
        this.QuestionsTable.tableArn + '/*',
      ]
    }))

    this.ServiceLambda.addToRolePolicy(new iam.PolicyStatement({
      sid: 'IVSAccess',
      effect: iam.Effect.ALLOW,
      actions: [
        "ivs:CreateChannel",
        "ivs:GetChannel",
        "ivs:GetStreamKey",
        "ivs:DeleteChannel",
      ],
      resources: [
        '*'
      ]
    }))

    this.ServiceLambda.addToRolePolicy(new iam.PolicyStatement({
      sid: 'BRASSAccess',
      effect: iam.Effect.ALLOW,
      actions: [
        "brassservice:IsAuthorized",
        "brassservice:BatchIsAuthorized",
      ],
      resources: [
        '*'
      ]
    }))
  }
}
