import cdk = require('@aws-cdk/core');
import lambda = require('@aws-cdk/aws-lambda');
import ec2 = require('@aws-cdk/aws-ec2');
import iam = require('@aws-cdk/aws-iam')

export interface ParsnipUIProps {
  trivEAARN: string
}

export class ParsnipUI extends cdk.Construct {
  ParsnipUILambda: lambda.Function

  constructor(scope: cdk.Construct, id: string, vpc: ec2.IVpc, apiLambda: lambda.IFunction, props: ParsnipUIProps) {
    super(scope, id);

    this.ParsnipUILambda = new lambda.Function(this, 'ParsnipUILambda', {
      vpc: vpc,
      runtime: lambda.Runtime.GO_1_X,
      handler: 'bin/parsnip-ui/handler',
      code: lambda.Code.asset('../bin/parsnip-ui.zip'),
      memorySize: 512,
      timeout: cdk.Duration.seconds(30),
      environment: {
        "parsnipAPILambdaARN": apiLambda.functionArn,
      }
    })

    this.ParsnipUILambda.addToRolePolicy(new iam.PolicyStatement({
      effect: iam.Effect.ALLOW,
      actions: [
        "ssm:GetParameter",
        "ssm:GetParameters",
      ],
      resources: [
        cdk.Arn.format({
          partition: "aws",
          service: "ssm",
          account: cdk.Stack.of(this).account,
          region: cdk.Stack.of(this).region,
          resource: 'parameter',
          resourceName: 'parsnip_csrfKey', // This created OOB and added manually to SSM
        }, cdk.Stack.of(this))
      ]
    }));

    this.ParsnipUILambda.addToRolePolicy(new iam.PolicyStatement({
      effect: iam.Effect.ALLOW,
      actions: ["lambda:InvokeFunction"],
      resources: ["*"],
    }))

    this.ParsnipUILambda.addPermission("PermissionForTrivEAToInvokeLambda", {
      principal: new iam.ServicePrincipal("apigateway.amazonaws.com"),
      sourceArn: props.trivEAARN,
      action: "lambda:InvokeFunction",
    })
  }
}
