package main

import (
	"fmt"
	"net/http"
	"os"
	"strconv"
	"time"

	"code.justin.tv/amzn/TwirpGoLangAWSTransports/lambda"
	parsnip "code.justin.tv/event-engineering/parsnip/pkg/rpc"
	"code.justin.tv/event-engineering/parsnip/pkg/svc"
	"code.justin.tv/video/brassclient"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/session"
	"github.com/aws/aws-sdk-go/service/dynamodb"
	"github.com/aws/aws-sdk-go/service/ivs"
	"github.com/aws/aws-sdk-go/service/ssm"
	"github.com/sirupsen/logrus"
)

func main() {
	logger := logrus.New()
	logger.SetLevel(logrus.DebugLevel)

	channelsTableName := os.Getenv("channelsTableName")
	questionsTableName := os.Getenv("questionsTableName")
	publisherBindleLockID := os.Getenv("publisherBindleLockID")
	adminBindleLockID := os.Getenv("adminBindleLockID")
	playbackAuthKeyParameterName := os.Getenv("playbackAuthKeyParameterName")
	acao := os.Getenv("acao")
	tokenExpirySecondsStr := os.Getenv("tokenExpirySeconds")

	tokenExpirySeconds, err := strconv.ParseInt(tokenExpirySecondsStr, 10, 64)
	if err != nil {
		panic(fmt.Errorf("Failed to parse token expiry %v", err))
	}

	// Create am AWS session
	sess, err := session.NewSession(&aws.Config{
		Region: aws.String(os.Getenv("AWS_REGION")),
	})

	if err != nil {
		panic(fmt.Errorf("Error creating AWS session %v", err))
	}

	ddbClient := dynamodb.New(sess)
	ivsClient := ivs.New(sess)
	ssmClient := ssm.New(sess)
	brassClient := brassclient.New(sess)

	// Create the Service
	parsnipServer, err := svc.New(channelsTableName, questionsTableName, playbackAuthKeyParameterName, time.Second*time.Duration(tokenExpirySeconds), acao, ddbClient, ivsClient, ssmClient, brassClient, svc.BindleLockConfig{
		IsAdminBindleLockID:     adminBindleLockID,
		IsPublisherBindleLockID: publisherBindleLockID,
	}, logger)

	if err != nil {
		panic(err)
	}

	// Create a Twirp service based off the internal service
	err = lambda.ListenAndServe(":80", &headerMapper{
		baseHandler: parsnip.NewParsnipServer(parsnipServer, nil),
	})

	if err != nil {
		// TODO logging and metrics
		panic(err)
	}
}

// This will map certain http headers to context values so we can get to them inside our service
type headerMapper struct {
	baseHandler http.Handler
}

func (hm *headerMapper) ServeHTTP(w http.ResponseWriter, req *http.Request) {
	ctx := req.Context()

	userIdentifier := req.Header.Get(svc.UserIDHeader)
	ctx = svc.WithUserID(ctx, userIdentifier)

	req = req.WithContext(ctx)

	hm.baseHandler.ServeHTTP(w, req)
}
