package main

import (
	"encoding/json"
	"fmt"
	"log"
	"net/http"
	"os"
	"sync"
	"time"

	"code.justin.tv/event-engineering/parsnip/app"
	"code.justin.tv/event-engineering/parsnip/app/api/auth"
	parsnipSvc "code.justin.tv/event-engineering/parsnip/pkg/svc"
	"code.justin.tv/jleroux/go-isengard2/service"
	"code.justin.tv/jleroux/go-isengard2/service/isengard"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/credentials"
	"github.com/aws/aws-sdk-go/aws/session"
	"github.com/aws/aws-sdk-go/service/dynamodb"
	"github.com/aws/aws-sdk-go/service/ivs"
	"github.com/aws/aws-sdk-go/service/ssm"
	"github.com/sirupsen/logrus"
	goji "goji.io"
)

// This is dev config only. Staging/Prod config is set up via enviromnent variables in CDK
var (
	parsnipAPIAccountID    = "282758828945" // parsnip+stg@amazon.com
	parsnipAPIIsengardRole = "Admin"        // Realistically we should set up Dev roles here that are linked to an IAM role we can use to configure permissions properly but... meh
	parsnipAPIRegion       = "us-west-2"
	parsnipAPILambdaARN    = ""
)

type helloResp struct {
	User string `json:"user"`
}

func main() {
	var wg sync.WaitGroup

	logger := logrus.New()
	logger.SetLevel(logrus.DebugLevel)

	// Create a new isengard client to establish who the user is and generate credentials necessary to contact services
	svc, err := service.NewIsengardService()
	if err != nil {
		fmt.Println("Failed to initiate isengard client (you probably need to run mwinit)")
		logger.Trace(err)
		os.Exit(1)
	}

	// Grab user identity with Hello
	r, err := svc.Hello(&isengard.HelloInput{})
	if err != nil {
		fmt.Println("Failed to establish developer identity (you probably need to run mwinit)")
		logger.Trace(err)
		os.Exit(1)
	}

	hr := &helloResp{}
	err = json.Unmarshal([]byte(*r.Message), hr)
	if err != nil {
		fmt.Printf("Failed to decode developer identity: %v\n", err)
		os.Exit(1)
	}

	sess, err := session.NewSession(&aws.Config{
		Region: aws.String(parsnipAPIRegion),
		Credentials: credentials.NewCredentials(&isengardCredentialsProvider{
			svc:       svc,
			accountID: parsnipAPIAccountID,
			role:      parsnipAPIIsengardRole,
		}),
	})

	if err != nil {
		fmt.Printf("Failed to generate credentials for CarrotControl: %v\n", err)
		os.Exit(1)
	}

	// Create a dummy auth middleware for development using the identity that we've established via isengard
	rootMux := goji.NewMux()

	rootMux.Use(func(inner http.Handler) http.Handler {
		return auth.WithDummyAuth(inner, hr.User)
	})

	/* For Working locally */
	ddbClient := dynamodb.New(sess)
	ivsClient := ivs.New(sess)
	ssmClient := ssm.New(sess)

	// Create the Service
	parsnipServer, err := parsnipSvc.New("parsnip-channels", "parsnip-questions", "parsnip_playbackKey", time.Second*10, "https://localhost:3000", ddbClient, ivsClient, ssmClient, nil, parsnipSvc.BindleLockConfig{}, logger)
	if err != nil {
		fmt.Printf("Failed to generate parsnip service: %v\n", err)
		os.Exit(1)
	}

	app.New(logger, parsnipServer, rootMux, "")

	// For using staging lambda
	/*
		parsnipAPI := parsnip.NewParsnipProtobufClient("https://would.you.like.a.carrot.twitch.tv", twirpLambda.NewClient(lambda.New(sess), parsnipAPILambdaARN))
		app.New(logger, parsnipAPI, rootMux, "")
	*/

	fmt.Printf("Starting Dev Server as user %v\n", hr.User)
	wg.Add(1)
	go func() {
		defer wg.Done()

		err := http.ListenAndServe(":8337", rootMux)
		if err != nil {
			log.Println(err)
		}
	}()

	wg.Wait()
}

type isengardCredentialsProvider struct {
	svc        *isengard.Isengard
	accountID  string
	role       string
	expiration int64
}

func (icp *isengardCredentialsProvider) Retrieve() (credentials.Value, error) {
	output, err := icp.svc.GetAssumeRoleCredentials(&isengard.GetAssumeRoleCredentialsInput{
		AWSAccountID: aws.String(icp.accountID),
		IAMRoleName:  aws.String(icp.role),
	})

	if err != nil {
		return credentials.Value{}, err
	}

	result := &service.AssumeRoleResult{}
	err = json.Unmarshal([]byte(*output.AssumeRoleResult), result)
	if err != nil {
		return credentials.Value{}, err
	}

	icp.expiration = result.Credentials.Expiration

	return credentials.Value{
		AccessKeyID:     result.Credentials.AccessKeyID,
		SecretAccessKey: result.Credentials.SecretAccessKey,
		SessionToken:    result.Credentials.SessionToken,
		ProviderName:    "isengardCredentialsProvider",
	}, nil
}

func (icp *isengardCredentialsProvider) IsExpired() bool {
	return time.Now().Unix() >= icp.expiration
}
