package main

import (
	"fmt"
	"net/http"
	"os"

	twirpLambda "code.justin.tv/amzn/TwirpGoLangAWSTransports/lambda"
	"code.justin.tv/event-engineering/parsnip/app"
	"code.justin.tv/event-engineering/parsnip/app/api/auth"
	pc "code.justin.tv/event-engineering/parsnip/pkg/client"
	parsnip "code.justin.tv/event-engineering/parsnip/pkg/rpc"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/session"
	"github.com/aws/aws-sdk-go/service/lambda"
	"github.com/aws/aws-sdk-go/service/ssm"
	"github.com/sirupsen/logrus"
	goji "goji.io"
)

func main() {
	logger := logrus.New()
	logger.SetLevel(logrus.DebugLevel)

	parsnipAPILambdaARN := os.Getenv("parsnipAPILambdaARN")

	// Create am AWS session
	sess, err := session.NewSession(&aws.Config{
		Region: aws.String(os.Getenv("AWS_REGION")),
	})

	if err != nil {
		panic(fmt.Errorf("Error creating AWS session %v", err))
	}

	// Auth all requests via midway
	rootMux := goji.NewMux()
	rootMux.Use(func(inner http.Handler) http.Handler {
		return auth.WithAmznSSO(inner)
	})

	// Set up parsnip client
	parsnipAPI := parsnip.NewParsnipProtobufClient("https://would.you.like.a.carrot.twitch.tv", pc.New(twirpLambda.NewClient(lambda.New(sess), parsnipAPILambdaARN)))

	// Grab csrf key from SSM
	ssmClient := ssm.New(sess)

	csrfKeyParam, err := ssmClient.GetParameter(&ssm.GetParameterInput{
		Name: aws.String("parsnip_csrfKey"),
	})

	if err != nil {
		panic(err)
	}

	app.New(logger, parsnipAPI, rootMux, *csrfKeyParam.Parameter.Value)

	err = twirpLambda.ListenAndServe(":80", rootMux)
	if err != nil {
		// TODO logging and metrics
		panic(err)
	}
}
