package svc

import (
	"context"
	"regexp"

	parsnip "code.justin.tv/event-engineering/parsnip/pkg/rpc"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/service/dynamodb"
	"github.com/aws/aws-sdk-go/service/dynamodb/dynamodbattribute"
	"github.com/aws/aws-sdk-go/service/ivs"
	"github.com/twitchtv/twirp"
)

var validChannelRegex = regexp.MustCompile("^[a-zA-Z0-9-_]+$")

func (c *client) CreateChannel(ctx context.Context, request *parsnip.CreateChannelRequest) (*parsnip.ChannelInfo, error) {
	isAdmin, isPublisher, err := c.getPermissions(ctx)
	if err != nil {
		c.logger.WithError(err).Warn("Failed to establish permissions")
		return nil, err
	}

	if !isAdmin && !isPublisher {
		return nil, twirp.NewError(twirp.PermissionDenied, "You do not have permission to perform this action")
	}

	if !validChannelRegex.MatchString(request.ChannelId) {
		return nil, twirp.InvalidArgumentError("ChannelId", "Channel IDs can only contain letters, numbers, hyphens and underscores")
	}

	ivsResp, err := c.ivs.CreateChannel(&ivs.CreateChannelInput{
		Name:       aws.String(request.ChannelId),
		Authorized: aws.Bool(true),
	})

	if err != nil {
		c.logger.WithError(err).Warnf("Failed to create channel in IVS")
		return nil, twirp.InternalError("Failed to create channel")
	}

	channelIDHash := getChannelIDHash(request.ChannelId)
	ddbc := &ddbChannel{
		ChannelIDHash:      channelIDHash,
		ChannelID:          request.ChannelId,
		IVSChannelARN:      *ivsResp.Channel.Arn,
		IVSStreamKeyARN:    *ivsResp.StreamKey.Arn,
		ChannelTitle:       request.ChannelTitle,
		ChannelDescription: request.ChannelDescription,
	}

	item, err := dynamodbattribute.MarshalMap(ddbc)

	if err != nil {
		c.logger.WithError(err).Warnf("Failed to marshal ddbChannel")

		_, delErr := c.ivs.DeleteChannel(&ivs.DeleteChannelInput{
			Arn: ivsResp.Channel.Arn,
		})

		if delErr != nil {
			c.logger.WithError(err).Warnf("Failed to delete IVS channel after marshal error")
		}

		return nil, twirp.InternalError("Failed to create channel")
	}

	_, err = c.ddb.PutItem(&dynamodb.PutItemInput{
		TableName:           aws.String(c.channelsTableName),
		Item:                item,
		ConditionExpression: aws.String("attribute_not_exists(channel_id_hash)"),
	})

	if err != nil {
		c.logger.WithError(err).Warn("Failed to write channel to dynamodb")

		_, delErr := c.ivs.DeleteChannel(&ivs.DeleteChannelInput{
			Arn: ivsResp.Channel.Arn,
		})

		if delErr != nil {
			c.logger.WithError(err).Warnf("Failed to delete IVS channel after PutItem error")
		}

		return nil, twirp.InternalError("Failed to create channel")
	}

	return &parsnip.ChannelInfo{
		ChannelIdHash:      channelIDHash,
		ChannelId:          request.ChannelId,
		IvsChannelArn:      *ivsResp.Channel.Arn,
		IvsPlaybackUrl:     *ivsResp.Channel.PlaybackUrl,
		IvsStreamKey:       *ivsResp.StreamKey.Value,
		ChannelTitle:       request.ChannelTitle,
		ChannelDescription: request.ChannelDescription,
	}, nil
}
