package svc

import (
	"context"

	parsnip "code.justin.tv/event-engineering/parsnip/pkg/rpc"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/service/dynamodb"
	"github.com/aws/aws-sdk-go/service/dynamodb/dynamodbattribute"
	"github.com/aws/aws-sdk-go/service/ivs"
	"github.com/twitchtv/twirp"
)

func (c *client) DeleteChannel(ctx context.Context, request *parsnip.DeleteChannelRequest) (*parsnip.DeleteChannelResponse, error) {
	isAdmin, _, err := c.getPermissions(ctx)
	if err != nil {
		c.logger.WithError(err).Warn("Failed to establish permissions")
		return nil, err
	}

	if !isAdmin {
		return nil, twirp.NewError(twirp.PermissionDenied, "You do not have permission to perform this action")
	}

	item, err := c.ddbGetChannel(request.ChannelIdHash)

	if err != nil {
		return nil, err
	}

	channel, err := c.hydrateChannel(item)

	if err != nil {
		return nil, err
	}

	_, err = c.ivs.DeleteChannel(&ivs.DeleteChannelInput{
		Arn: aws.String(channel.IvsChannelArn),
	})

	if err != nil {
		c.logger.WithError(err).Warnf("Failed to delete channel in IVS")
		return nil, twirp.InternalError("Failed to delete channel")
	}

	idAttr, err := dynamodbattribute.Marshal(request.ChannelIdHash)
	if err != nil {
		c.logger.WithError(err).Warnf("Failed to generate ID attribute")
		return nil, twirp.InternalError("Failed to delete channel")
	}

	_, err = c.ddb.DeleteItem(&dynamodb.DeleteItemInput{
		TableName: aws.String(c.channelsTableName),
		Key:       map[string]*dynamodb.AttributeValue{"channel_id_hash": idAttr},
	})

	if err != nil {
		c.logger.WithError(err).Warn("Failed to delete channel from dynamodb")
		return nil, twirp.InternalError("Failed to delete channel")
	}

	return &parsnip.DeleteChannelResponse{}, nil
}
