package svc

import (
	"context"
	"errors"

	parsnip "code.justin.tv/event-engineering/parsnip/pkg/rpc"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/service/dynamodb"
	"github.com/aws/aws-sdk-go/service/dynamodb/dynamodbattribute"
	"github.com/aws/aws-sdk-go/service/ivs"
	"github.com/twitchtv/twirp"
)

func (c *client) GetChannel(ctx context.Context, request *parsnip.GetChannelRequest) (*parsnip.ChannelInfo, error) {
	isAdmin, isPublisher, err := c.getPermissions(ctx)
	if err != nil {
		c.logger.WithError(err).Warn("Failed to establish permissions")
		return nil, err
	}

	if !isAdmin && !isPublisher {
		return nil, twirp.NewError(twirp.PermissionDenied, "You do not have permission to perform this action")
	}

	item, err := c.ddbGetChannel(request.ChannelIdHash)

	if err != nil {
		return nil, err
	}

	channel, err := c.hydrateChannel(item)

	if err != nil {
		// TODO: Work out how to do better error handling depending on the type of error that occurs
		c.logger.WithError(err).Warnf("Failed to retrieve channel")
		return nil, twirp.InternalError("Failed to retrieve channel")
	}

	return channel, nil
}

func (c *client) ddbGetChannel(channelIDHash string) (map[string]*dynamodb.AttributeValue, error) {
	idAttr, err := dynamodbattribute.Marshal(channelIDHash)
	if err != nil {
		return nil, err
	}

	resp, err := c.ddb.GetItem(&dynamodb.GetItemInput{
		TableName: aws.String(c.channelsTableName),
		Key: map[string]*dynamodb.AttributeValue{
			"channel_id_hash": idAttr,
		},
	})

	if err != nil {
		return nil, err
	}

	return resp.Item, nil
}

// TODO: Figure out why the fuck I built it like this rather than just hashing the ID and looking it up like that...
func (c *client) ddbGetChannelByID(channelID string) (map[string]*dynamodb.AttributeValue, error) {
	idAttr, err := dynamodbattribute.Marshal(channelID)
	if err != nil {
		return nil, err
	}

	resp, err := c.ddb.Query(&dynamodb.QueryInput{
		TableName:              aws.String(c.channelsTableName),
		IndexName:              aws.String("idx_by_channel_id"),
		KeyConditionExpression: aws.String("channel_id = :channel_id"),
		ExpressionAttributeValues: map[string]*dynamodb.AttributeValue{
			":channel_id": idAttr,
		},
	})

	if err != nil {
		return nil, err
	}

	if len(resp.Items) != 1 {
		return nil, errors.New("Channel Not Found")
	}

	return resp.Items[0], nil
}

func (c *client) hydrateChannel(item map[string]*dynamodb.AttributeValue) (*parsnip.ChannelInfo, error) {
	var ddbc ddbChannel
	err := dynamodbattribute.ConvertFromMap(item, &ddbc)

	if err != nil {
		return nil, err
	}

	// Get channel info from IVS
	ivsResp, err := c.ivs.GetChannel(&ivs.GetChannelInput{
		Arn: aws.String(ddbc.IVSChannelARN),
	})

	if err != nil {
		return nil, err
	}

	skResp, err := c.ivs.GetStreamKey(&ivs.GetStreamKeyInput{
		Arn: aws.String(ddbc.IVSStreamKeyARN),
	})

	if err != nil {
		return nil, err
	}

	return &parsnip.ChannelInfo{
		ChannelIdHash:      ddbc.ChannelIDHash,
		ChannelId:          ddbc.ChannelID,
		ChannelTitle:       ddbc.ChannelTitle,
		ChannelDescription: ddbc.ChannelDescription,
		BindleLockId:       ddbc.BindleLockID,
		IvsChannelArn:      ddbc.IVSChannelARN,
		IvsPlaybackUrl:     *ivsResp.Channel.PlaybackUrl,
		IvsStreamKey:       *skResp.StreamKey.Value,
		IvsIngestUrl:       *ivsResp.Channel.IngestEndpoint,
	}, nil
}
